/*
    This file is part of ethereum.js.

    ethereum.js is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    ethereum.js is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with ethereum.js.  If not, see <http://www.gnu.org/licenses/>.
*/
/** @file contract.js
 * @authors:
 *   Marek Kotewicz <marek@ethdev.com>
 * @date 2014
 */

var web3 = require('./web3'); 
var abi = require('./abi');
var eventImpl = require('./event');

var addFunctionRelatedPropertiesToContract = function (contract) {
    
    contract.call = function (options) {
        contract._isTransact = false;
        contract._options = options;
        return contract;
    };

    contract.transact = function (options) {
        contract._isTransact = true;
        contract._options = options;
        return contract;
    };

    contract._options = {};
    ['gas', 'gasPrice', 'value', 'from'].forEach(function(p) {
        contract[p] = function (v) {
            contract._options[p] = v;
            return contract;
        };
    });

};

var addFunctionsToContract = function (contract, desc, address) {
    var inputParser = abi.inputParser(desc);
    var outputParser = abi.outputParser(desc);

    // create contract functions
    abi.filterFunctions(desc).forEach(function (method) {

        var displayName = abi.methodDisplayName(method.name);
        var typeName = abi.methodTypeName(method.name);

        var impl = function () {
            var params = Array.prototype.slice.call(arguments);
            var signature = abi.methodSignature(method.name);
            var parsed = inputParser[displayName][typeName].apply(null, params);

            var options = contract._options || {};
            options.to = address;
            options.data = signature + parsed;
            
            var isTransact = contract._isTransact === true || (contract._isTransact !== false && !method.constant);
            var collapse = options.collapse !== false;
            
            // reset
            contract._options = {};
            contract._isTransact = null;

            if (isTransact) {
                // it's used byt natspec.js
                // TODO: figure out better way to solve this
                web3._currentContractAbi = desc;
                web3._currentContractAddress = address;
                web3._currentContractMethodName = method.name;
                web3._currentContractMethodParams = params;

                // transactions do not have any output, cause we do not know, when they will be processed
                web3.eth.transact(options);
                return;
            }
            
            var output = web3.eth.call(options);
            var ret = outputParser[displayName][typeName](output);
            if (collapse)
            {
                if (ret.length === 1)
                    ret = ret[0];
                else if (ret.length === 0)
                    ret = null;
            }
            return ret;
        };

        if (contract[displayName] === undefined) {
            contract[displayName] = impl;
        }

        contract[displayName][typeName] = impl;
    });
};

var addEventRelatedPropertiesToContract = function (contract, desc, address) {
    contract.address = address;
    
    Object.defineProperty(contract, 'topic', {
        get: function() {
            return abi.filterEvents(desc).map(function (e) {
                return abi.methodSignature(e.name);
            });
        }
    });

};

var addEventsToContract = function (contract, desc, address) {
    // create contract events
    abi.filterEvents(desc).forEach(function (e) {

        var impl = function () {
            var params = Array.prototype.slice.call(arguments);
            var signature = abi.methodSignature(e.name);
            var event = eventImpl(address, signature, e);
            var o = event.apply(null, params);
            return web3.eth.watch(o);  
        };
        
        // this property should be used by eth.filter to check if object is an event
        impl._isEvent = true;

        // TODO: we can remove address && topic properties, they are not used anymore since we introduced _isEvent
        impl.address = address;

        Object.defineProperty(impl, 'topic', {
            get: function() {
                return [abi.methodSignature(e.name)];
            }
        });
        
        // TODO: rename these methods, cause they are used not only for methods
        var displayName = abi.methodDisplayName(e.name);
        var typeName = abi.methodTypeName(e.name);

        if (contract[displayName] === undefined) {
            contract[displayName] = impl;
        }

        contract[displayName][typeName] = impl;

    });
};


/**
 * This method should be called when we want to call / transact some solidity method from javascript
 * it returns an object which has same methods available as solidity contract description
 * usage example: 
 *
 * var abi = [{
 *      name: 'myMethod',
 *      inputs: [{ name: 'a', type: 'string' }],
 *      outputs: [{name: 'd', type: 'string' }]
 * }];  // contract abi
 *
 * var myContract = web3.eth.contract('0x0123123121', abi); // creation of contract object
 *
 * myContract.myMethod('this is test string param for call'); // myMethod call (implicit, default)
 * myContract.call().myMethod('this is test string param for call'); // myMethod call (explicit)
 * myContract.transact().myMethod('this is test string param for transact'); // myMethod transact
 *
 * @param address - address of the contract, which should be called
 * @param desc - abi json description of the contract, which is being created
 * @returns contract object
 */

var contract = function (address, desc) {

    // workaround for invalid assumption that method.name is the full anonymous prototype of the method.
    // it's not. it's just the name. the rest of the code assumes it's actually the anonymous
    // prototype, so we make it so as a workaround.
    // TODO: we may not want to modify input params, maybe use copy instead?
    desc.forEach(function (method) {
        if (method.name.indexOf('(') === -1) {
            var displayName = method.name;
            var typeName = method.inputs.map(function(i){return i.type; }).join();
            method.name = displayName + '(' + typeName + ')';
        }
    });

    var result = {};
    addFunctionRelatedPropertiesToContract(result);
    addFunctionsToContract(result, desc, address);
    addEventRelatedPropertiesToContract(result, desc, address);
    addEventsToContract(result, desc, address);

    return result;
};

module.exports = contract;

