package main

import (
	"fmt"
	"github.com/ethereum/eth-go"
	"github.com/ethereum/eth-go/ethutil"
	"github.com/ethereum/go-ethereum/ethereal/ui"
	"github.com/ethereum/go-ethereum/utils"
	"github.com/go-qml/qml"
	"github.com/rakyll/globalconf"
	"log"
	"os"
	"os/signal"
	"path"
	"runtime"
)

const Debug = true

// Register interrupt handlers so we can stop the ethereum
func RegisterInterupts(s *eth.Ethereum) {
	// Buffered chan of one is enough
	c := make(chan os.Signal, 1)
	// Notify about interrupts for now
	signal.Notify(c, os.Interrupt)
	go func() {
		for sig := range c {
			fmt.Printf("Shutting down (%v) ... \n", sig)

			s.Stop()
		}
	}()
}

func main() {
	Init()

	qml.Init(nil)

	runtime.GOMAXPROCS(runtime.NumCPU())

	g, err := globalconf.NewWithOptions(&globalconf.Options{
		Filename: path.Join(ethutil.ApplicationFolder(Datadir), "conf.ini"),
	})
	if err != nil {
		fmt.Println(err)
	} else {
		g.ParseAll()
	}
	ethutil.ReadConfig(Datadir, ethutil.LogFile|ethutil.LogStd, g, Identifier)

	// Instantiated a eth stack
	ethereum, err := eth.New(eth.CapDefault, UseUPnP)
	if err != nil {
		log.Println("eth start err:", err)
		return
	}
	ethereum.Port = OutboundPort

	if GenAddr {
		fmt.Println("This action overwrites your old private key. Are you sure? (y/n)")

		var r string
		fmt.Scanln(&r)
		for ; ; fmt.Scanln(&r) {
			if r == "n" || r == "y" {
				break
			} else {
				fmt.Printf("Yes or no?", r)
			}
		}

		if r == "y" {
			utils.CreateKeyPair(true)
		}
		os.Exit(0)
	} else {
		if len(ImportKey) > 0 {
			fmt.Println("This action overwrites your old private key. Are you sure? (y/n)")
			var r string
			fmt.Scanln(&r)
			for ; ; fmt.Scanln(&r) {
				if r == "n" || r == "y" {
					break
				} else {
					fmt.Printf("Yes or no?", r)
				}
			}

			if r == "y" {
				utils.ImportPrivateKey(ImportKey)
				os.Exit(0)
			}
		}
	}

	if ExportKey {
		keyPair := ethutil.GetKeyRing().Get(0)
		fmt.Printf(`
Generating new address and keypair.
Please keep your keys somewhere save.

++++++++++++++++ KeyRing +++++++++++++++++++
addr: %x
prvk: %x
pubk: %x
++++++++++++++++++++++++++++++++++++++++++++
save these words so you can restore your account later: %s
`, keyPair.Address(), keyPair.PrivateKey, keyPair.PublicKey)

		os.Exit(0)
	}

	if ShowGenesis {
		fmt.Println(ethereum.BlockChain().Genesis())
		os.Exit(0)
	}

	/*
		if StartMining {
			utils.DoMining(ethereum)
		}
	*/

	if StartRpc {
		utils.DoRpc(ethereum, RpcPort)
	}

	log.Printf("Starting Ethereum GUI v%s\n", ethutil.Config.Ver)

	// Set the max peers
	ethereum.MaxPeers = MaxPeer

	gui := ethui.New(ethereum)

	ethereum.Start(UseSeed)

	gui.Start(AssetPath)

	// Wait for shutdown
	ethereum.WaitForShutdown()
}
