package basicpiece

import (
	"bytes"
	"encoding/binary"

	"golang.org/x/xerrors"

	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/cid"
)

type DecodedData struct {
	HasPre  bool
	PrePieceCommit cid.Commit

	Data []byte

	PieceCommit []cid.Commit
	CommitData  []byte
}

// DecodedData called piece follows as:
// [CommLen|DataLen|Data]...[CommLen|DataLen|Data|Commit][HasPre&CommLen|DataLen|PreCommit|Commit]
//                                         🢙🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢘🢛
func (data *DecodedData) Serialize() ([]byte, error) {
	var buf []byte
	MetaLen := uint32(len(data.Data))
	CommLen := uint32(len(data.CommitData))
	if data.HasPre {
		if MetaLen > 0 {
			return nil, xerrors.Errorf("")
		}
		buf = make([]byte, nextUppandedPowerOfTwo(40+CommLen))
		binary.BigEndian.PutUint32(buf[:4], 0x80000000)
		binary.BigEndian.PutUint32(buf[4:8], CommLen)
		copy(buf[8:40], data.PrePieceCommit[:])
		copy(buf[40:], data.CommitData[:])
	} else {
		buf = make([]byte, nextUppandedPowerOfTwo(8+MetaLen+CommLen))
		binary.BigEndian.PutUint32(buf[:4], MetaLen)
		binary.BigEndian.PutUint32(buf[4:8], CommLen)
		copy(buf[8:8+MetaLen], data.Data[:])
		copy(buf[8+MetaLen:], data.CommitData[:])
	}
	return buf, nil
}

func (data *DecodedData) Deserialize(buf []byte) error {
	var err error
	var MetaLen uint32
	var CommLen uint32

	read := len(buf)
	if read < 8 {
		return xerrors.Errorf("can't deserialize the data less then 8bytes")
	}

	binary.Read(bytes.NewReader(buf[0:4]), binary.BigEndian, &MetaLen)
	binary.Read(bytes.NewReader(buf[4:8]), binary.BigEndian, &CommLen)
	data.HasPre = (MetaLen >> 31) != 0
	MetaLen = MetaLen & 0x7fffffff
	rbuf := buf[8:read]
	if data.HasPre {
		if read < 40 {
			return xerrors.Errorf("can't read the pre-piece-hash")
		}
		copy(data.PrePieceCommit[:], buf[8:40])
		rbuf = rbuf[32:]
	}

	if uint32(len(rbuf)) <= MetaLen {
		data.Data = rbuf[:]
	} else if uint32(len(rbuf)) <= CommLen+MetaLen {
		data.Data = rbuf[:MetaLen]
		data.PieceCommit, err = to32ByteCommit(rbuf[MetaLen:])
		if err != nil {
			return err
		}
	} else {
		data.Data = rbuf[:MetaLen]
		data.PieceCommit, err = to32ByteCommit(rbuf[MetaLen : CommLen+MetaLen])
		if err != nil {
			return err
		}
	}
	return nil
}

func to32ByteCommit(in []byte) ([]cid.Commit, error) {
	if len(in)%32 != 0 {
		return nil, xerrors.Errorf("lenth of the hash arr must be multiple of 32")
	}
	hash := make([]cid.Commit, len(in)/32)
	for index := 0; index < len(hash); index++ {
		copy(hash[index][:], in[index*32:index*32+32])
	}
	return hash, nil
}

func nextUppandedPowerOfTwo(index uint32) abi.UnpaddedPieceSize {
	index--
	power := 0
	for index = index / 254; index != 0; power += 1 {
		index >>= 1
	}
	return abi.UnpaddedPieceSize(254 * (1 << power))
}