package seal

import (
	"context"

	"github.com/minio/blake2b-simd"

	"fil_integrate/build/cid"
	spproof "fil_integrate/build/proof"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/build/storiface"
	"fil_integrate/seal/basicfs"
	"fil_integrate/seal/basicpiece"
)

var b = blake2b.Sum256([]byte("randomness"))
var Ticket abi.SealRandomness = abi.SealRandomness(b)

type PieceEncoder interface {
	// Split and encode data into pieces
	// Pieces structure is [ Tag | MetaData | HashData ] or [ Tag | PreHash | HashData]
	EncodeDataToPieces(ctx context.Context, sectorSize abi.SectorSize, file storage.Data) (abi.PieceInfo, []abi.PieceInfo, error)
	LoadPiece(ctx context.Context, pieceID cid.Commit) ([]byte, error)
	DecodePiece(ctx context.Context, buf []byte) (*basicpiece.DecodedData, error)
}

//interface
type SectorSealer interface {
	SavePiece(ctx context.Context, piece abi.PieceInfo, in storage.Data, size int) error

	AddPiece(ctx context.Context, sector storage.SectorRef, existingPieceSizes []abi.UnpaddedPieceSize, piece abi.PieceInfo) error
	// run pre-commit1 and pre-commit2 phase
	// generate the sealed sector and sector commitment(commd, commr)
	Sealed(ctx context.Context, sid storage.SectorRef, pieces []abi.PieceInfo) (storage.SectorCids, error)
	// run commit1 and commit2 phase
	// generate the zk-proof of sealing
	GenerateCommitProof(ctx context.Context, sid storage.SectorRef, seed abi.InteractiveSealRandomness, pieces []abi.PieceInfo, cids storage.SectorCids) (spproof.Proof, error)
	AggregateSealProofs(aggregateInfo spproof.AggregateSealVerifyProofAndInfos, proofs []spproof.Proof) (spproof.Proof, error)

	ReadPiece(ctx context.Context, pieceID cid.Commit) ([]byte, error)
	UnsealedPieceRange(ctx context.Context, sid storage.SectorRef, unsealed cid.Commit, offset abi.UnpaddedByteIndex, size abi.UnpaddedPieceSize) ([]byte, error)

	GenerateWindowPoStProofs(ctx context.Context, minerID abi.ActorID, sectorInfo []spproof.SectorInfo, randomness abi.PoStRandomness) (spproof.PoStProof, []abi.SectorID, error)
	AggregateWindowPoStProofs(aggregateInfo spproof.AggregateWindowPostInfos, proofs []spproof.PoStProof) (spproof.PoStProof, error)
}

type SectorVerifier interface {
	VerifySeal(info spproof.SealVerifyInfo) (bool, error)
	VerifyAggregateSeals(aggregate spproof.AggregateSealVerifyProofAndInfos) (bool, error)

	VerifyWindowPoSt(info spproof.WindowPoStVerifyInfo) (bool, error)
	VerifyAggregateWindowPostProofs(aggregate spproof.AggregateWindowPostInfos) (bool, error)
	VerifyPieceAndDataRoot(proofType abi.RegisteredSealProof, commd cid.Commit, pieces []abi.PieceInfo) (bool, error)
}

type SectorManager interface {
	GetRoot() string
	// * returns storiface.ErrSectorNotFound if a requested existing sector doesn't exist
	// * returns an error when allocate is set, and existing isn't, and the sector exists
	AcquireSector(ctx context.Context, id storage.SectorRef, existing storiface.SectorFileType, allocate storiface.SectorFileType) (storiface.SectorPaths, func(), error)
	AcquirePiece(ctx context.Context, id cid.Commit, existing storiface.SectorFileType, allocate storiface.SectorFileType) (storiface.SectorPaths, func(), error)
}

var _ SectorManager = &basicfs.Manager{}
