package pieces

import (
	"golang.org/x/xerrors"
	"github.com/minio/sha256-simd"

	"fil_integrate/build/fr32"
	"fil_integrate/build/state-types/abi"
)

func nextPowerOfTwo(v abi.PaddedPieceSize) abi.PaddedPieceSize {
	v--
	v |= v >> 1
	v |= v >> 2
	v |= v >> 4
	v |= v >> 8
	v |= v >> 16
	v++
	return v
}

func GeneratePieceCommitmentFast(data []byte, unpad abi.UnpaddedPieceSize) ([32]byte, error) {
	var result [32]byte

	err := unpad.Validate()
	if err != nil {
		return [32]byte{}, err
	}

	padSize := unpad.Padded()
	out := make([]byte, padSize)

	fr32.Pad(data, out)

	// r, err := MerkleTreeRecurse(out)
	r, err := MerkleTreeLoop(out)
	if err != nil {
		return [32]byte{}, err
	}
	copy(result[:], r)
	return result, nil
}

func MerkleTreeRecurse(data []byte) ([]byte, error) {
	n := uint64(len(data))
	//叶结点，直接返回
	if n < 32 {
		return data, xerrors.Errorf("can not generate the merkle tree")
	}
	if n == 32 {
		return data, nil
	}
	k := len(data) / 2
	h := sha256.New()
	//求左子树
	x, err := MerkleTreeRecurse(data[0:k])
	if err != nil {
		return nil, err
	}
	// 修剪到fr域
	trim_to_fr32(x)
	h.Write(x[:])
	//求右子树
	x, err = MerkleTreeRecurse(data[k:n])
	if err != nil {
		return nil, err
	}
	trim_to_fr32(x)
	h.Write(x[:])
	//得到哈希结果
	res := h.Sum(nil)
	trim_to_fr32(res)
	return res, nil
}

func MerkleTreeLoop(data []byte) ([]byte, error) {
	n := abi.PaddedPieceSize(len(data))

	if n != nextPowerOfTwo(n) {
		return nil, xerrors.Errorf("can not generate the merkle tree")
	}
	for lenth := abi.PaddedPieceSize(32); lenth < n; lenth <<= 1 {
		for index := abi.PaddedPieceSize(0); index < n; {
			windex := index
			h := sha256.New()

			// write left child
			trim_to_fr32(data[index : index+32])
			h.Write(data[index : index+32])
			index += lenth
			// write right child
			trim_to_fr32(data[index : index+32])
			h.Write(data[index : index+32])
			index += lenth

			res := h.Sum(nil)
			copy(data[windex:windex+32], res)
		}
	}
	trim_to_fr32(data[:32])
	return data[:32], nil
}

func trim_to_fr32(data []byte) {
	// strip last two bits, to ensure result is in Fr.
	data[31] &= 0b0011_1111
}
