package pieces

import (
	"github.com/minio/sha256-simd"
	"golang.org/x/xerrors"

	"fil_integrate/build/cid"
	"fil_integrate/build/fr32"
	"fil_integrate/build/state-types/abi"
)

type MerkleTree struct {
	leaves abi.PaddedPieceSize
	// merkle tree data except leaves node
	data []cid.Commit
	root cid.Commit
}

func GenerateMerkleTree(data []byte, unpad abi.UnpaddedPieceSize) (MerkleTree, error) {
	err := unpad.Validate()
	if err != nil {
		return MerkleTree{}, err
	}

	padSize := unpad.Padded()
	out := make([]byte, padSize)

	fr32.Pad(data, out)

	return MerkleTreeBuild(out)
}

func MerkleTreeBuild(data []byte) (MerkleTree, error) {
	n := abi.PaddedPieceSize(len(data))
	if n != nextPowerOfTwo(n) {
		return MerkleTree{}, xerrors.Errorf("can not generate the merkle tree")
	}
	var tree MerkleTree
	tree.leaves = n / 32
	tree.data = make([]cid.Commit, tree.leaves-1)
	data_index := 0

	for index := abi.PaddedPieceSize(0); index < n; data_index++ {
		h := sha256.New()

		// write left child
		trim_to_fr32(data[index : index+32])
		h.Write(data[index : index+32])
		index += 32
		// write right child
		trim_to_fr32(data[index : index+32])
		h.Write(data[index : index+32])
		index += 32

		res := h.Sum(nil)
		trim_to_fr32(res[:32])
		copy(tree.data[data_index][:], res)
	}

	for child := 0; data_index > child+1; data_index++ {
		h := sha256.New()

		// write left child
		h.Write(tree.data[child][:])
		child++
		// write left child
		h.Write(tree.data[child][:])
		child++

		res := h.Sum(nil)
		trim_to_fr32(res[:32])
		copy(tree.data[data_index][:], res)
	}

	tree.root = tree.data[len(tree.data)-1]
	return tree, nil
}

func (tree MerkleTree) GetRoot() cid.Commit {
	return tree.root
}

func (tree MerkleTree) GenProof(offset abi.PaddedPieceSize, size abi.PaddedPieceSize) (MerkleProof, error) {
	// we can't gengerate the merkle proof from leaves,
	// because the merkle tree don't store the meta data
	if size != nextPowerOfTwo(size) || offset % size != 0 || size <= 32 {
		return MerkleProof{}, xerrors.Errorf("can not generate the merkle proof")
	}
	offset /= 32
	size /= 32

	// begin at level 2
	width := tree.leaves >> 1
	size >>= 1
	offset >>= 1

	base := abi.PaddedPieceSize(0)
	// find the sub-tree(offset, size) index at merkle-tree
	for size > 1 {
		base += width
		size >>= 1
		offset >>= 1
		width >>= 1
	}

	var proof *MerkleProof = &MerkleProof{}
	for base < tree.leaves-2 {
		index := (offset >> 1) << 1
		if index != offset {
			proof.Add(tree.data[base + index], LEFT)
		} else {
			proof.Add(tree.data[base + index + 1], RIGHT)
		}
		base += width
		offset >>= 1
		width >>= 1
	}
	return *proof, nil
}
