package user

import (
	"context"
	"io"

	"golang.org/x/xerrors"

	"fil_integrate/actor/connect"
	"fil_integrate/build/cid"
	spieces "fil_integrate/build/pieces"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/seal"
	"fil_integrate/seal/basicpiece"
)

type User struct {
	sectorSize abi.SectorSize
	encoder    seal.PieceEncoder
	conn       *connect.Connection
	pieceChan  chan abi.PieceInfo
	doneChan   chan struct{}
	cid2sidMap map[cid.Commit]abi.ActorID
}

var _ UserAPI = &User{}

func New(encoder seal.PieceEncoder, conn *connect.Connection) *User {
	u := &User{
		sectorSize: abi.SectorSize(storage.SectorSize32MiB),
		cid2sidMap: make(map[cid.Commit]abi.ActorID),
		pieceChan:  make(chan abi.PieceInfo, 32),
		doneChan:   make(chan struct{}),
		encoder:    encoder,
		conn:       conn,
	}
	return u
}

func (u *User) EncodeDataToPieces(ctx context.Context, file storage.Data) (abi.PieceInfo, error) {
	finalPiece, pieces, err := u.encoder.EncodeDataToPieces(ctx, u.sectorSize, file)
	// send piece to provider
	for _, piece := range pieces {
		u.pieceChan <- piece
	}
	return finalPiece, err
}

func (u *User) SendPiecesToMiner(ctx context.Context, miner abi.ActorID, size abi.PaddedPieceSize) error {
	for{
		select {
		case piece := <- u.pieceChan:
			err := u.handleSend(ctx, miner, piece, size)
			if err != nil {
				return err
			}
		case <- u.doneChan:
			Loop:
				for{
					select{
					case piece := <- u.pieceChan:
						err := u.handleSend(ctx, miner, piece, size)
						if err != nil {
							return err
						}
					default:
						break Loop
					}
				}
			err := u.conn.SendEncodeDone(ctx)
			if err != nil {
				return err
			}
			log.Infof("send pieces done")
			return nil
		case <- ctx.Done():
			return xerrors.Errorf("context canceled")
		}
	}
	return nil
}

func (u *User) handleSend(ctx context.Context, miner abi.ActorID, piece abi.PieceInfo, size abi.PaddedPieceSize) error {
	buf, err := u.encoder.LoadPiece(ctx, piece.PieceCID)
	if err != nil {
		return err
	}

	log.Infof("sending piece")

	tree, err := spieces.GenerateMerkleTree(buf, abi.UnpaddedPieceSize(len(buf)))
	if err != nil {
		return err
	}

	if size > abi.UnpaddedPieceSize(len(buf)).Padded() {
		size = abi.UnpaddedPieceSize(len(buf)).Padded()
	}
	
	for offset := abi.PaddedPieceSize(0) ; offset < piece.Size; offset += size {
		data := buf[:size.Unpadded()]
		buf = buf[size.Unpadded():]

		mproof, err := tree.GenProof(offset, size)
		if err != nil {
			return err
		}

		ok, err := mproof.Verify(data[:], piece.PieceCID, offset)
		if err != nil {
			return err
		}
		if !ok {
			return xerrors.Errorf("merkle proof verify failed")
		}

		err = u.conn.SendPieceToProvider(ctx, data, mproof, piece)
		if err != nil {
			return err
		}
	}
	return nil
}

func (u *User) SendPiecesDone(ctx context.Context) error {
	select{
	case u.doneChan <- struct{}{}:
		break
	case <- ctx.Done():
		return xerrors.Errorf("context canceled")
	}
	return nil
}

func (u *User) ReadPieceRange(
	ctx context.Context,
	out io.Writer,
	piece abi.PieceInfo,
	offset uint64,
	size uint64,
) error {
	log.Infof("Reading Piece [%d:%d]", offset, offset+size)
	UnpaddedSectorSize := abi.PaddedPieceSize(u.sectorSize).Unpadded()
	DataLen := uint32(UnpaddedSectorSize) - seal.TagLen

	data, err := u.getPiece(ctx, piece)
	if err != nil {
		return err
	}
	piecesCommit := data.PieceCommit

	for data.HasPre {
		data, err = u.getPiece(ctx, abi.PieceInfo{
			Size: abi.PaddedPieceSize(storage.SectorSize32MiB),
			PieceCID: data.PrePieceCommit,
		})
		if err != nil {
			return err
		}
		piecesCommit = append(data.PieceCommit, piecesCommit...)
	}
	buf := data.Data[:]
	maxSize := uint64(len(piecesCommit))*uint64(DataLen) + uint64(len(buf))

	if offset == 0 && size == 0 {
		size = maxSize
	}
	if size+offset > maxSize {
		return xerrors.Errorf("Piece Size is Out of Range [offset: %w, size:%w, max_size:%w]", offset, size, maxSize)
	}

	piecesCommit = piecesCommit[offset/uint64(DataLen):]
	rangePiece := &RangePiece{
		offset: offset,
		size:   size,
		lenth:  uint64(DataLen),
	}
	for {
		rstart, rsize := rangePiece.nextRange()
		if rsize == 0 {
			break
		}

		var wbuf []byte
		if len(piecesCommit) != 0 {
			data, err := u.getPiece(ctx, abi.PieceInfo{
				Size: abi.PaddedPieceSize(storage.SectorSize32MiB),
				PieceCID: piecesCommit[0],
			})
			if err != nil {
				return err
			}

			wbuf = data.Data[rstart:]
			piecesCommit = piecesCommit[1:]
		} else {
			wbuf = buf[rstart:]
		}

		_, err = out.Write(wbuf[:rsize])
		if err != nil {
			return err
		}
	}

	return nil
}

func (u *User) getPiece(ctx context.Context, piece abi.PieceInfo) (*basicpiece.DecodedData, error) {
	// todo: GET from chian/provider
	// miner, ok := cid2sidMap[pieceCommit]
	buf, err := u.GetPieceFromProvider(ctx, 10000, piece)
	if err != nil {
		return nil, err
	}
	return u.encoder.DecodePiece(ctx, buf)
}

func (u *User) GetPieceFromProvider(ctx context.Context, miner abi.ActorID, piece abi.PieceInfo) ([]byte, error) {
	var buf = make([]byte, piece.Size.Unpadded())
	err := u.conn.RequestPiece(ctx, piece.PieceCID)
	if err != nil {
		return nil, err
	}

	var offset abi.PaddedPieceSize = 0
	for offset < piece.Size {
		op, data, err := u.conn.P2UMessage(ctx)
		if err != nil {
			return nil, err
		}
		if op != connect.OP_SEND_PIECE {
			return nil, xerrors.Errorf("Unexpected operator")
		}
		pieceInfo := data.(connect.PieceInfo)

		ok, err := (&pieceInfo.Proof).Verify(pieceInfo.Data, piece.PieceCID, offset)
		if err != nil {
			return nil, err
		}
		if !ok {
			return nil, xerrors.Errorf("merkle proof verify failed")
		}

		copy(buf[offset.Unpadded():], pieceInfo.Data)
		offset += abi.UnpaddedPieceSize(len(pieceInfo.Data)).Padded()
	}
	return buf, nil
}

type RangePiece struct {
	offset uint64
	size   uint64
	lenth  uint64
}

func (rp *RangePiece) nextRange() (uint64, uint64) {
	start := rp.offset % rp.lenth
	size := min(rp.size, rp.lenth-start)
	rp.offset = rp.offset + size
	rp.size = rp.size - size
	return start, size
}

func min(x, y uint64) uint64 {
	if x < y {
		return x
	}
	return y
}
