package connect

import (
	"context"
	"fmt"
	"sync"
	"time"

	"golang.org/x/xerrors"

	spieces "fil_integrate/build/pieces"
	spproof "fil_integrate/build/proof"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
)

func TestConnection() error {
	conn := NewConnection()
	ctx := context.Background()
	// Create a new context, with its cancellation function
	// from the original context
	ctx, cancel := context.WithCancel(ctx)
	var wg sync.WaitGroup
	wg.Add(3)
	go func() {
		defer wg.Done()
		err := runUser(conn, ctx)
		if err != nil {
			cancel()
			fmt.Printf("User: %+v\n", err)
		}
		fmt.Println("User: done")
	}()
	go func() {
		defer wg.Done()
		err := runProvider(conn, ctx)
		if err != nil {
			cancel()
			fmt.Printf("Provider: %+v\n", err)
		}
		fmt.Println("Provider: done")
	}()
	go func() {
		defer wg.Done()
		err := runKeeper(conn, ctx)
		if err != nil {
			cancel()
			fmt.Printf("Keeper: %+v\n", err)
		}
		fmt.Println("Keeper: done")
	}()
	wg.Wait()
	return nil
}

func runUser(conn *Connection, ctx context.Context) error {
	numFile := 4
	for i := 0; i < numFile; i++ {
		fmt.Println("User: Addding Piece")
		time.Sleep(2 * time.Second)
		conn.SendPieceToProvider(ctx, nil, spieces.MerkleProof{}, abi.PieceInfo{})
	}
	for i := 0; i < numFile; i++ {
		op, data, err := conn.U2PMessage(ctx)
		if err != nil {
			return err
		}
		if op != OP_SEND_PIECE {
			return xerrors.Errorf("unsupported operator")
		}
		fmt.Println("User: Reading piece", data)
	}
	return nil
}

func runProvider(conn *Connection, ctx context.Context) error {
	numFile := 4
	for i := 0; i < numFile; i++ {
		op, data, err := conn.U2PMessage(ctx)
		if err != nil {
			return err
		}
		if op != OP_SEND_PIECE {
			return xerrors.Errorf("unsupported operator")
		}
		fmt.Println("Provider: save piece", data)
	}
	for sum := 10; sum > 0; {
		for i := 0; i < 4; i++ {
			fmt.Println("Provider: sealing sector")
			conn.RequestSealRandom(ctx, abi.SectorID{})
			op, data, err := conn.P2KMessage(ctx)
			if err != nil {
				return err
			}
			if op != OP_SEND_SEAL_RANDOM {
				return xerrors.Errorf("unsupported operator")
			}
			fmt.Println("Provider: Generating Commit Proof", data)
			conn.SendSealProof(ctx, nil, abi.SectorID{}, storage.SectorCids{})
			sum--
			if sum <= 0 {
				break
			}
		}
		fmt.Println("Provider: Aggregating Commit Proof")
		conn.SendAggregateSealProof(ctx, nil, nil, nil)

		fmt.Println("Provider: Generating Window PoSt")
		conn.SendWindowPoStProof(ctx, spproof.PoStProof{}, abi.PoStRandomness{}, 0)
	}
	fmt.Println("Provider: Aggregating Window PoSt")
	conn.SendAggregateWindowPoStProof(ctx, spproof.PoStProof{}, nil, 0)
	for i := 0; i < numFile; i++ {
		conn.SendPieceToUser(ctx, nil, spieces.MerkleProof{}, abi.PieceInfo{})
	}
	return nil
}

func runKeeper(conn *Connection, ctx context.Context) error {
	for {
		op, data, err := conn.P2KMessage(ctx)
		if err != nil {
			return err
		}
		switch op {
		case OP_REQUEST_SEAL_RANDOM:
			fmt.Println("Keeper: Generating random seed", data)
			conn.SendSealRandom(ctx, abi.InteractiveSealRandomness{})
		case OP_REQUEST_POST_RANDOM:
			fmt.Println("Keeper: Generating random seed", data)
			conn.SendPoStRandom(ctx, abi.PoStRandomness{})
		case OP_SEND_SEAL_PROOF:
			fmt.Println("Keeper: Verifying seal proof")
		case OP_SEND_AGGREGATE_SEAL_PROOF:
			fmt.Println("Keeper: verifying aggregate seal proof", data)
		case OP_SEND_WINDOW_POST_PROOF:
			fmt.Println("Keeper: verifying window post proof", data)
		case OP_SEND_AGGREGATE_WINDOW_POST_PROOF:
			fmt.Println("Keeper: verifying aggregate window post proof", data)
			return nil
		default:
			return xerrors.Errorf("unsupported operator")
		}
	}
	return nil
}

// func hadleUserMessage(conn *Connection, op Operator, data interface{}) error {
// 	switch op {
// 	case OP_REQUEST_PIECE:
// 		fmt.Println("Provider: send piece to user")
// 		conn.SendPiece(ctx, nil, cid.Commit{})
// 	case OP_SEND_PIECE:
// 		fmt.Println("Provider: save piece")
// 		fmt.Println("Provider: sealing sector")
// 		conn.RequestRandom(ctx, storage.SectorRef{})
// 	default:
// 		return xerrors.Errorf("unsupported operator")
// 	}
// 	return nil
// }

// func hadleKeeperMessage(conn *Connection, op Operator, data interface{}) error {
// 	switch op{
// 	case OP_SEND_RANDOM:
// 		fmt.Println("Provider: generating commit proof")
// 		conn.SendSealProof(ctx, nil, storage.SectorRef{}, cid.Commit{}, cid.Commit{})
// 	case OP_REQUEST_WINDOW_POST:
// 		fmt.Println("Provider: generating window post")
// 		conn.SendWindowPoStProof(ctx, spproof.PoStProof{}, abi.PoStRandomness{}, 0)
// 	default:
// 		return xerrors.Errorf("unsupported operator")
// 	}
// 	return nil
// }
