package connect

import (
	"context"

	"golang.org/x/xerrors"

	"fil_integrate/build/cid"
	spieces "fil_integrate/build/pieces"
	spproof "fil_integrate/build/proof"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
)

type Connection struct {
	u2pChannel chan Data
	p2uChannel chan Data
	p2kChannel chan Data
}

func NewConnection() *Connection {
	return &Connection{
		u2pChannel: make(chan Data),
		p2uChannel: make(chan Data),
		p2kChannel: make(chan Data),
	}
}

func (conn *Connection) RequestPiece(ctx context.Context, pieceCommit cid.Commit) error {
	sdata := Data{
		op:   OP_REQUEST_PIECE,
		data: pieceCommit,
	}
	select {
	case conn.u2pChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendPieceToUser(ctx context.Context, data []byte, proof spieces.MerkleProof, piece abi.PieceInfo) error {
	buf := make([]byte, len(data))
	copy(buf, data)
	sdata := Data{
		op: OP_SEND_PIECE,
		data: PieceInfo{
			Data:  buf,
			Proof: proof,
			Piece: piece,
		},
	}
	select {
	case conn.p2uChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendPieceToProvider(ctx context.Context, data []byte, proof spieces.MerkleProof, piece abi.PieceInfo) error {
	buf := make([]byte, len(data))
	copy(buf, data)
	sdata := Data{
		op: OP_SEND_PIECE,
		data: PieceInfo{
			Data:  buf,
			Proof: proof,
			Piece: piece,
		},
	}
	select {
	case conn.u2pChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendEncodeDone(ctx context.Context) error {
	sdata := Data{
		op:   OP_ENCODE_DONE,
		data: nil,
	}
	select {
	case conn.u2pChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendSealDone(ctx context.Context) error {
	sdata := Data{
		op:   OP_SEAL_DONE,
		data: nil,
	}
	select {
	case conn.p2uChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) RequestSealRandom(ctx context.Context, sid abi.SectorID) error {
	sdata := Data{
		op:   OP_REQUEST_SEAL_RANDOM,
		data: sid,
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) RequestPoStRandom(ctx context.Context, sids []abi.SectorID) error {
	var buf []abi.SectorID
	buf = append(buf, sids...)
	sdata := Data{
		op:   OP_REQUEST_POST_RANDOM,
		data: buf,
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendSealRandom(ctx context.Context, random abi.InteractiveSealRandomness) error {
	sdata := Data{
		op:   OP_SEND_SEAL_RANDOM,
		data: random,
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendPoStRandom(ctx context.Context, random abi.PoStRandomness) error {
	sdata := Data{
		op:   OP_SEND_POST_RANDOM,
		data: random,
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendSealProof(
	ctx context.Context,
	proof spproof.Proof,
	sid abi.SectorID,
	commit storage.SectorCids,
) error {
	var buf spproof.Proof
	buf = append(buf, proof...)
	sdata := Data{
		op: OP_SEND_SEAL_PROOF,
		data: SealProofInfo{
			Proof:  buf,
			Sector: sid,
			Commit: commit,
		},
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendAggregateSealProof(
	ctx context.Context,
	proof spproof.Proof,
	sids []abi.SectorID,
	commits []storage.SectorCids,
) error {
	var buf spproof.Proof
	buf = append(buf, proof...)
	sdata := Data{
		op: OP_SEND_AGGREGATE_SEAL_PROOF,
		data: AggregateSealProofInfo{
			Proof:   buf,
			Sectors: sids,
			Commits: commits,
		},
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendWindowPoStProof(
	ctx context.Context,
	proof spproof.PoStProof,
	randomness abi.PoStRandomness,
	miner abi.ActorID,
) error {
	var buf spproof.Proof
	buf1 := spproof.PoStProof{
		ProofBytes: append(buf, proof.ProofBytes...),
		PoStProof: proof.PoStProof,
	}
	sdata := Data{
		op: OP_SEND_WINDOW_POST_PROOF,
		data: WindowPoStProofInfo{
			Proof:      buf1,
			Randomness: randomness,
			Miner:      miner,
		},
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) SendAggregateWindowPoStProof(
	ctx context.Context,
	proof spproof.PoStProof,
	randomnesses []abi.PoStRandomness,
	miner abi.ActorID,
) error {
	var buf spproof.Proof
	buf1 := spproof.PoStProof{
		ProofBytes: append(buf, proof.ProofBytes...),
		PoStProof: proof.PoStProof,
	}
	buf2 := make([]abi.PoStRandomness, len(randomnesses))
	for i, random := range randomnesses {
		buf2[i] = random
	}
	sdata := Data{
		op: OP_SEND_AGGREGATE_WINDOW_POST_PROOF,
		data: AggregateWindowPoStProofInfo{
			Proof:        buf1,
			Randomnesses: buf2,
			Miner:        miner,
		},
	}
	select {
	case conn.p2kChannel <- sdata:
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
	return nil
}

func (conn *Connection) U2PMessage(ctx context.Context) (Operator, interface{}, error) {
	select {
	case mess := <-conn.u2pChannel:
		return mess.op, mess.data, nil
	case <-ctx.Done():
		if conn.u2pChannel != nil {
			close(conn.u2pChannel)
			conn.u2pChannel = nil
		}
		return OP_CLOSED, nil, xerrors.Errorf("context canceled")
	}
}

func (conn *Connection) P2UMessage(ctx context.Context) (Operator, interface{}, error) {
	select {
	case mess := <-conn.p2uChannel:
		return mess.op, mess.data, nil
	case <-ctx.Done():
		if conn.p2uChannel != nil {
			close(conn.p2uChannel)
			conn.p2uChannel = nil
		}
		return OP_CLOSED, nil, xerrors.Errorf("context canceled")
	}
}

func (conn *Connection) P2KMessage(ctx context.Context) (Operator, interface{}, error) {
	select {
	case mess := <-conn.p2kChannel:
		return mess.op, mess.data, nil
	case <-ctx.Done():
		if conn.p2kChannel != nil {
			close(conn.p2kChannel)
			conn.p2kChannel = nil
		}
		return OP_CLOSED, nil, xerrors.Errorf("context canceled")
	}
}
