package multisig

import (
	"bytes"
	"encoding/binary"

	adt4 "github.com/filecoin-project/specs-actors/v4/actors/util/adt"

	"github.com/filecoin-project/go-address"
	"github.com/filecoin-project/go-state-types/abi"
	"github.com/ipfs/go-cid"
	cbg "github.com/whyrusleeping/cbor-gen"
	"golang.org/x/xerrors"

	"github.com/filecoin-project/lotus/chain/actors/adt"

	builtin4 "github.com/filecoin-project/specs-actors/v4/actors/builtin"

	msig4 "github.com/filecoin-project/specs-actors/v4/actors/builtin/multisig"
)

var _ State = (*state4)(nil)

func load4(store adt.Store, root cid.Cid) (State, error) {
	out := state4{store: store}
	err := store.Get(store.Context(), root, &out)
	if err != nil {
		return nil, err
	}
	return &out, nil
}

func make4(store adt.Store, signers []address.Address, threshold uint64, startEpoch abi.ChainEpoch, unlockDuration abi.ChainEpoch, initialBalance abi.TokenAmount) (State, error) {
	out := state4{store: store}
	out.State = msig4.State{}
	out.State.Signers = signers
	out.State.NumApprovalsThreshold = threshold
	out.State.StartEpoch = startEpoch
	out.State.UnlockDuration = unlockDuration
	out.State.InitialBalance = initialBalance

	em, err := adt4.StoreEmptyMap(store, builtin4.DefaultHamtBitwidth)
	if err != nil {
		return nil, err
	}

	out.State.PendingTxns = em

	return &out, nil
}

type state4 struct {
	msig4.State
	store adt.Store
}

func (s *state4) LockedBalance(currEpoch abi.ChainEpoch) (abi.TokenAmount, error) {
	return s.State.AmountLocked(currEpoch - s.State.StartEpoch), nil
}

func (s *state4) StartEpoch() (abi.ChainEpoch, error) {
	return s.State.StartEpoch, nil
}

func (s *state4) UnlockDuration() (abi.ChainEpoch, error) {
	return s.State.UnlockDuration, nil
}

func (s *state4) InitialBalance() (abi.TokenAmount, error) {
	return s.State.InitialBalance, nil
}

func (s *state4) Threshold() (uint64, error) {
	return s.State.NumApprovalsThreshold, nil
}

func (s *state4) Signers() ([]address.Address, error) {
	return s.State.Signers, nil
}

func (s *state4) ForEachPendingTxn(cb func(id int64, txn Transaction) error) error {
	arr, err := adt4.AsMap(s.store, s.State.PendingTxns, builtin4.DefaultHamtBitwidth)
	if err != nil {
		return err
	}
	var out msig4.Transaction
	return arr.ForEach(&out, func(key string) error {
		txid, n := binary.Varint([]byte(key))
		if n <= 0 {
			return xerrors.Errorf("invalid pending transaction key: %v", key)
		}
		return cb(txid, (Transaction)(out)) //nolint:unconvert
	})
}

func (s *state4) PendingTxnChanged(other State) (bool, error) {
	other4, ok := other.(*state4)
	if !ok {
		// treat an upgrade as a change, always
		return true, nil
	}
	return !s.State.PendingTxns.Equals(other4.PendingTxns), nil
}

func (s *state4) transactions() (adt.Map, error) {
	return adt4.AsMap(s.store, s.PendingTxns, builtin4.DefaultHamtBitwidth)
}

func (s *state4) decodeTransaction(val *cbg.Deferred) (Transaction, error) {
	var tx msig4.Transaction
	if err := tx.UnmarshalCBOR(bytes.NewReader(val.Raw)); err != nil {
		return Transaction{}, err
	}
	return tx, nil
}

func (s *state4) GetState() interface{} {
	return &s.State
}
