package seal

import(
	"context"
	"io"
	"os"
	// "bytes"

	"golang.org/x/xerrors"

	proof5 "github.com/filecoin-project/specs-actors/v5/actors/runtime/proof"
	logging "github.com/ipfs/go-log/v2"
	// commcid "github.com/filecoin-project/go-fil-commcid"
	ffi "github.com/filecoin-project/filecoin-ffi"
	"fil_integrate/extern/sector-storage/ffiwrapper"
	"fil_integrate/extern/sector-storage/ffiwrapper/basicfs"
	"fil_integrate/extern/sector-storage/storiface"

	"github.com/filecoin-project/specs-storage/storage"
	"github.com/filecoin-project/go-state-types/abi"
	"github.com/filecoin-project/go-state-types/network"
	"github.com/ipfs/go-cid"
)

var log = logging.Logger("sealing")

//32字节，总共256位
//[has_pre][MetaLen1..MetaLen4][PieceLen1..PieceLen4]
const TagLen int = 8

const arp = abi.RegisteredAggregationProof_SnarkPackV1

const NewestNetworkVersion = network.Version13

func DefaultAggregationType() abi.RegisteredAggregationProof {
	return arp;
}

func GetCommRFromDir(
	sb *ffiwrapper.Sealer,
	ctx context.Context,
	sectorID abi.SectorID,
) (cid.Cid, error) {
	return sb.GetCommRFromDir(ctx, sectorID)
}

func PutCommRIntoDir(
	sb *ffiwrapper.Sealer,
	ctx context.Context,
	sectorID abi.SectorID,
	sealedCID cid.Cid,
) error {
	return sb.PutCommRIntoDir(ctx, sectorID, sealedCID)
}

func AddPiece(
	sb *ffiwrapper.Sealer, 
	ctx context.Context, 
	sector storage.SectorRef, 
	existingPieceSizes []abi.UnpaddedPieceSize, 
	pieceSize abi.UnpaddedPieceSize, 
	file storage.Data,
) (abi.PieceInfo, error) {

	return sb.AddPiece(ctx, sector, existingPieceSizes, pieceSize, file)

}

// 没有测试
// func EncodeDataToPieces(sb *ffiwrapper.Sealer, ctx context.Context, sector storage.SectorRef, sectorSize abi.SectorSize, file storage.Data) (abi.PieceInfo, error) {
// 	var piecesID []byte
// 	var FinalPiece abi.PieceInfo
// 	var pieceNumber int32 = 1

// 	UnpaddedSectorSize := abi.PaddedPieceSize(sectorSize).Unpadded()
// 	buf := make([]byte, UnpaddedSectorSize)

// 	DataLen := UnpaddedSectorSize-TagLen

// 	for{
// 		copy(buf[:TagLen], 0)
// 		var MetaLen int = 0
// 		var rerr error
// 		for rbuf := buf[TagLen:]; len(rbuf) > 0; {
// 			n, rerr := file.read(buf[TagLen:])

// 			if rerr != nil && rerr != io.EOF{
// 				return nil, rerr
// 			}
// 			rbuf = rbuf[n:]
// 			MetaLen += n

// 			if rerr == io.EOF{
// 				break
// 			}
// 		}
// 		if rerr == io.EOF{
// 			//encode first sector
// 			pieceNumber, FinalPiece, err = EncodeData(buf, MetaLen, DataLen, piecesID, pieceNumber)
// 			if err != nil{
// 				return nil, err
// 			}
// 			break
// 		}
// 		copy(buf[:4], []byte(MetaLen))

// 		piece, err := sb.AddPiece(ctx, sector, nil, UnpaddedSectorSize, bytes.NewReader(buf), pieceNumber)
// 		if err != nil{
// 			return nil, err
// 		}
// 		piecesID = append(pieces, commcid.CIDToReplicaCommitmentV1(piece.PieceCID)...)
// 		pieceNumber += 1
// 	}
// 	return pieceNumber, FinalPiece, nil
// }

// func EncodeData(buf []byte, MetaLen int, DataLen int, piecesID []byte, pieceNumber int32) (int32, abi.PieceInfo, error) {
// 	//encode first sector
// 	remain := len(piecesID)
// 	CommLen := min(remain, ((DataLen-MetaLen)/32) * 32)
// 	rbuf := buf[MetaLen+TagLen:]
	
//     copy(buf[:4], []byte(MetaLen))
// 	copy(buf[4:8], []byte(CommLen))
// 	copy(rbuf, piecesID[:CommLen])
// 	copy(rbuf[CommLen:], 0)

// 	piecesID = piecesID[CommLen:]
// 	MetaLen = 0
// 	remain -= CommLen
// 	rbuf = buf[TagLen+32:]

// 	prePiece, err := sb.AddPiece(ctx, sector, nil, UnpaddedSectorSize, bytes.NewReader(buf), pieceNumber)
// 	pieceNumber += 1
// 	if err != nil{
// 		return nil, err
// 	}

// 	for ;remain > 0; {
// 		//encode next n sector
// 		CommLen := min(remain, ((DataLen-32)/32) * 32)

// 		copy(buf[:4], []byte(MetaLen | 0x80000000))
// 		copy(buf[4:8], []byte(CommLen))
// 		copy(buf[8:40], commcid.CIDToReplicaCommitmentV1(prePiece.PieceCID)...)
// 		copy(rbuf, piecesID[:CommLen])
// 		copy(rbuf[CommLen:], 0)

// 		piecesID = piecesID[CommLen:]
// 		MetaLen = 0
// 		remain -= CommLen
// 		rbuf = buf[TagLen:]

// 		prePiece, err = sb.AddPiece(ctx, sector, nil, UnpaddedSectorSize, bytes.NewReader(buf), pieceNumber)
// 		pieceNumber += 1
// 		if err != nil{
// 			return nil, err
// 		}
// 	}

// 	return pieceNumber, prePiece, nil
// }

// func DecodePieceToData(sb *ffiwrapper.Sealer, ctx context.Context, out io.Writer, finalPiece abi.PieceInfo, pieceNum int32, sectorSize abi.SectorSize) ([]abi.PieceInfo, error){
// 	var piecesID []abi.PieceInfo
// 	var commds []byte

// 	unpaddedSectorSize := abi.PaddedPieceSize(sectorSize).Unpadded()
// 	buf := make([]byte, unpaddedSectorSize)
// 	DataLen := unpaddedSectorSize-TagLen

// 	for ;pieceNum > 0; PieceNum-- {
// 		ok, err := sb.ReadPiece(ctx, sid, bytes.NewWriter(buf), 0, unpaddedSectorSize, pieceNum)
// 		if err != nil {
// 			return nil
// 		}
// 		if !ok {
// 			return xerrors.Errorf("Read pieces error!")
// 		}

// 		MetaLen := buf[0:4] & 0x7fffffff
// 		HasPre := buf[0:4] >> 31
// 		CommLen := buf[4:8]
// 		rbuf := buf[8:]

// 		if HasPre {
// 			prePiece := buf[8:40]
// 			CommLen -= 32
// 			rbuf = rbuf[32:]
// 			commd, _ := commcid.DataCommitmentV1ToCID(prePiece)
// 			piecesID = append(piecesID, abi.PieceInfo{abi.PaddedPieceSize(sectorSize), commd})
// 		}
// 		data := rbuf[:MetaLen]
// 		commds = append(commds, rbuf[MetaLen:MetaLen+CommLen]...)
// 		//**data顺序错了，需要
// 		n, werr := out.write(data[:])
// 		if werr != nil {
// 			return nil, werr
// 		}
// 	}

// 	for cbuf := commds; len(cbuf) > 0; {
// 		commd, _ := cbuf.DataCommitmentV1ToCID(commds[32])
// 		piecesID = append(piecesID, abi.PieceInfo{abi.PaddedPieceSize(sectorSize), commd})
// 		cbuf = cbuf[32:]
// 	}
// 	return piecesID, nil
// }

func AggregateWindowPoStProofs(
	sb *ffiwrapper.Sealer, 
	ctx context.Context,
	aggregate abi.RegisteredAggregationProof, 
	randomnesses []abi.PoStRandomness,  
	windowPoStProofs []proof5.PoStProof,
	sectorCountArr []uint,
) ([]byte, error) {
	if len(windowPoStProofs) != len(sectorCountArr) {
		return nil, xerrors.Errorf("the lenth of windowPoStProofs and sectorCount is not match")
	}

	sectorCount := sectorCountArr[0]
	for _, count := range(sectorCountArr) {
		if sectorCount != count {
			return nil, xerrors.Errorf("Window PoSt challenge count must be equal")
		}
	}
	return sb.AggregateWindowPoStProofs(ctx, ffi.AggregateWindowPostInfos{
		Randomnesses: randomnesses,
		AggregateType: aggregate,
		Proofs: windowPoStProofs,
		SectorCount: sectorCount,
	})
}

func VerifyAggregateWindowPostProofs(
	ctx context.Context,
	proofType abi.RegisteredPoStProof,
	aggregateType abi.RegisteredAggregationProof,
	miner abi.ActorID,
	aggregateProof []byte,
	randomnesses []abi.PoStRandomness,
	sealedSectors [][]proof5.SectorInfo,
) (bool, error) {
	var sectorInfos []proof5.SectorInfo
	arr := make([]uint, len(sealedSectors))
	for i, sectors := range(sealedSectors) {
		arr[i] = uint(len(sectors))
		for _, sector := range(sectors) {
			sectorInfos = append(sectorInfos, sector)
		}
	}
	return ffiwrapper.ProofVerifier.VerifyAggregateWindowPostProofs(ctx, ffi.AggregateWindowPostInfos{
		PoStType: proofType,
		AggregateType: aggregateType,
		Miner: miner,
		AggregationProof: aggregateProof,
		Randomnesses: randomnesses,
		ChallengedSectors: sectorInfos,
		Arr: arr,
	})
}

func CheckPieceAndDataRoot(
	sid storage.SectorRef, 
	comm_d cid.Cid, 
	pieces []abi.PieceInfo,
) (bool, error) {
	UnsealedCID, err := ffiwrapper.GenerateUnsealedCID(sid.ProofType, pieces)
	if err != nil{
		return false, err
	}

	return comm_d == UnsealedCID, nil
}

func Sealed(
	sb *ffiwrapper.Sealer, 
	ctx context.Context, 
	sid storage.SectorRef, 
	seed abi.InteractiveSealRandomness, 
	ticket abi.SealRandomness, 
	pieces []abi.PieceInfo,
) (storage.SectorCids, storage.Proof, storage.Commit1Out, error) {
	// var sealedSectors saproof2.SectorInfo

	log.Infof("[%d] Running replication(1)...", sid.ID.Number)

	pc1out, err := sb.SealPreCommit1(ctx, sid, ticket, pieces)
	if err != nil {
		return storage.SectorCids{}, nil, nil, xerrors.Errorf("commit: %w", err)
	}

	log.Infof("[%d] Running replication(2)...", sid.ID.Number)
	cids, err := sb.SealPreCommit2(ctx, sid, pc1out)
	if err != nil {
		return storage.SectorCids{}, nil, nil, xerrors.Errorf("commit: %w", err)
	}

	log.Infof("[%d] Generating PoRep for sector (1)", sid.ID.Number)
	c1o, err := sb.SealCommit1(ctx, sid, ticket, seed, pieces, cids)
	if err != nil {
		return storage.SectorCids{}, nil, nil, err
	}

	log.Infof("[%d] Generating PoRep for sector (2)", sid.ID.Number)

	var proof storage.Proof
	proof, err = sb.SealCommit2(ctx, sid, c1o)
	if err != nil {
		return storage.SectorCids{}, nil, nil, err
	}

	return cids, proof, c1o, nil
}

func GenProofForC2(
	sb *ffiwrapper.Sealer, 
	ctx context.Context, 
	sid storage.SectorRef, 
	c1Out storage.Commit1Out,
) (storage.Proof, error) {
	return sb.SealCommit2(ctx, sid, c1Out)
}

func GenProofForWindowPoSt(
	sb *ffiwrapper.Sealer, 
	ctx context.Context, 
	minerID abi.ActorID, 
	sectorInfo []proof5.SectorInfo, 
	randomness abi.PoStRandomness,
) ([]proof5.PoStProof, []abi.SectorID, error) {
	return sb.GenerateWindowPoSt(ctx, minerID, sectorInfo, randomness)
}

func UnsealedRange(
	sb *ffiwrapper.Sealer, 
	ctx context.Context, 
	sbfs *basicfs.Provider, 
	sid storage.SectorRef, 
	sectorSize abi.SectorSize, 
	ticket abi.SealRandomness, 
	commd cid.Cid, 
	out io.Writer, 
	offset storiface.UnpaddedByteIndex, 
	size abi.UnpaddedPieceSize,
) error {
	log.Infof("[%d] Unsealing sector", sid.ID.Number)
	{
		p, done, err := sbfs.AcquireSector(ctx, sid, storiface.FTUnsealed, storiface.FTNone, storiface.PathSealing)
		if err != nil {
			return xerrors.Errorf("acquire unsealed sector for removing: %w", err)
		}
		done()

		if err := os.Remove(p.Unsealed); err != nil {
			return xerrors.Errorf("removing unsealed sector: %w", err)
		}
	}

	err := sb.UnsealPiece(ctx, sid, 0, abi.PaddedPieceSize(sectorSize).Unpadded(), ticket, commd)
	if err != nil {
		return err
	}

	ok, err := sb.ReadPiece(ctx, out, sid, offset, size)
	if err != nil{
		return err
	}

	if !ok {
		return xerrors.Errorf("Read pieces error!")
	}

	return nil

}