package paych

import (
	"encoding/base64"
	"fmt"

	"golang.org/x/xerrors"

	"github.com/filecoin-project/go-address"
	"github.com/filecoin-project/go-state-types/abi"
	big "github.com/filecoin-project/go-state-types/big"
	"github.com/filecoin-project/go-state-types/cbor"
	"github.com/ipfs/go-cid"
	ipldcbor "github.com/ipfs/go-ipld-cbor"

	paych0 "github.com/filecoin-project/specs-actors/actors/builtin/paych"

	builtin0 "github.com/filecoin-project/specs-actors/actors/builtin"

	builtin2 "github.com/filecoin-project/specs-actors/v2/actors/builtin"

	builtin3 "github.com/filecoin-project/specs-actors/v3/actors/builtin"

	builtin4 "github.com/filecoin-project/specs-actors/v4/actors/builtin"

	builtin5 "github.com/filecoin-project/specs-actors/v5/actors/builtin"

	"github.com/filecoin-project/lotus/chain/actors"
	"github.com/filecoin-project/lotus/chain/actors/adt"
	"github.com/filecoin-project/lotus/chain/actors/builtin"
	"github.com/filecoin-project/lotus/chain/types"
)

func init() {

	builtin.RegisterActorState(builtin0.PaymentChannelActorCodeID, func(store adt.Store, root cid.Cid) (cbor.Marshaler, error) {
		return load0(store, root)
	})

	builtin.RegisterActorState(builtin2.PaymentChannelActorCodeID, func(store adt.Store, root cid.Cid) (cbor.Marshaler, error) {
		return load2(store, root)
	})

	builtin.RegisterActorState(builtin3.PaymentChannelActorCodeID, func(store adt.Store, root cid.Cid) (cbor.Marshaler, error) {
		return load3(store, root)
	})

	builtin.RegisterActorState(builtin4.PaymentChannelActorCodeID, func(store adt.Store, root cid.Cid) (cbor.Marshaler, error) {
		return load4(store, root)
	})

	builtin.RegisterActorState(builtin5.PaymentChannelActorCodeID, func(store adt.Store, root cid.Cid) (cbor.Marshaler, error) {
		return load5(store, root)
	})
}

// Load returns an abstract copy of payment channel state, irregardless of actor version
func Load(store adt.Store, act *types.Actor) (State, error) {
	switch act.Code {

	case builtin0.PaymentChannelActorCodeID:
		return load0(store, act.Head)

	case builtin2.PaymentChannelActorCodeID:
		return load2(store, act.Head)

	case builtin3.PaymentChannelActorCodeID:
		return load3(store, act.Head)

	case builtin4.PaymentChannelActorCodeID:
		return load4(store, act.Head)

	case builtin5.PaymentChannelActorCodeID:
		return load5(store, act.Head)

	}
	return nil, xerrors.Errorf("unknown actor code %s", act.Code)
}

func MakeState(store adt.Store, av actors.Version) (State, error) {
	switch av {

	case actors.Version0:
		return make0(store)

	case actors.Version2:
		return make2(store)

	case actors.Version3:
		return make3(store)

	case actors.Version4:
		return make4(store)

	case actors.Version5:
		return make5(store)

	}
	return nil, xerrors.Errorf("unknown actor version %d", av)
}

func GetActorCodeID(av actors.Version) (cid.Cid, error) {
	switch av {

	case actors.Version0:
		return builtin0.PaymentChannelActorCodeID, nil

	case actors.Version2:
		return builtin2.PaymentChannelActorCodeID, nil

	case actors.Version3:
		return builtin3.PaymentChannelActorCodeID, nil

	case actors.Version4:
		return builtin4.PaymentChannelActorCodeID, nil

	case actors.Version5:
		return builtin5.PaymentChannelActorCodeID, nil

	}

	return cid.Undef, xerrors.Errorf("unknown actor version %d", av)
}

// State is an abstract version of payment channel state that works across
// versions
type State interface {
	cbor.Marshaler
	// Channel owner, who has funded the actor
	From() (address.Address, error)
	// Recipient of payouts from channel
	To() (address.Address, error)

	// Height at which the channel can be `Collected`
	SettlingAt() (abi.ChainEpoch, error)

	// Amount successfully redeemed through the payment channel, paid out on `Collect()`
	ToSend() (abi.TokenAmount, error)

	// Get total number of lanes
	LaneCount() (uint64, error)

	// Iterate lane states
	ForEachLaneState(cb func(idx uint64, dl LaneState) error) error

	GetState() interface{}
}

// LaneState is an abstract copy of the state of a single lane
type LaneState interface {
	Redeemed() (big.Int, error)
	Nonce() (uint64, error)
}

type SignedVoucher = paych0.SignedVoucher
type ModVerifyParams = paych0.ModVerifyParams

// DecodeSignedVoucher decodes base64 encoded signed voucher.
func DecodeSignedVoucher(s string) (*SignedVoucher, error) {
	data, err := base64.RawURLEncoding.DecodeString(s)
	if err != nil {
		return nil, err
	}

	var sv SignedVoucher
	if err := ipldcbor.DecodeInto(data, &sv); err != nil {
		return nil, err
	}

	return &sv, nil
}

var Methods = builtin5.MethodsPaych

func Message(version actors.Version, from address.Address) MessageBuilder {
	switch version {

	case actors.Version0:
		return message0{from}

	case actors.Version2:
		return message2{from}

	case actors.Version3:
		return message3{from}

	case actors.Version4:
		return message4{from}

	case actors.Version5:
		return message5{from}

	default:
		panic(fmt.Sprintf("unsupported actors version: %d", version))
	}
}

type MessageBuilder interface {
	Create(to address.Address, initialAmount abi.TokenAmount) (*types.Message, error)
	Update(paych address.Address, voucher *SignedVoucher, secret []byte) (*types.Message, error)
	Settle(paych address.Address) (*types.Message, error)
	Collect(paych address.Address) (*types.Message, error)
}
