package multisig

import (
	"bytes"
	"encoding/binary"

	adt0 "github.com/filecoin-project/specs-actors/actors/util/adt"

	"github.com/filecoin-project/go-address"
	"github.com/filecoin-project/go-state-types/abi"
	"github.com/ipfs/go-cid"
	cbg "github.com/whyrusleeping/cbor-gen"
	"golang.org/x/xerrors"

	"github.com/filecoin-project/lotus/chain/actors/adt"

	msig0 "github.com/filecoin-project/specs-actors/actors/builtin/multisig"
)

var _ State = (*state0)(nil)

func load0(store adt.Store, root cid.Cid) (State, error) {
	out := state0{store: store}
	err := store.Get(store.Context(), root, &out)
	if err != nil {
		return nil, err
	}
	return &out, nil
}

func make0(store adt.Store, signers []address.Address, threshold uint64, startEpoch abi.ChainEpoch, unlockDuration abi.ChainEpoch, initialBalance abi.TokenAmount) (State, error) {
	out := state0{store: store}
	out.State = msig0.State{}
	out.State.Signers = signers
	out.State.NumApprovalsThreshold = threshold
	out.State.StartEpoch = startEpoch
	out.State.UnlockDuration = unlockDuration
	out.State.InitialBalance = initialBalance

	em, err := adt0.MakeEmptyMap(store).Root()
	if err != nil {
		return nil, err
	}

	out.State.PendingTxns = em

	return &out, nil
}

type state0 struct {
	msig0.State
	store adt.Store
}

func (s *state0) LockedBalance(currEpoch abi.ChainEpoch) (abi.TokenAmount, error) {
	return s.State.AmountLocked(currEpoch - s.State.StartEpoch), nil
}

func (s *state0) StartEpoch() (abi.ChainEpoch, error) {
	return s.State.StartEpoch, nil
}

func (s *state0) UnlockDuration() (abi.ChainEpoch, error) {
	return s.State.UnlockDuration, nil
}

func (s *state0) InitialBalance() (abi.TokenAmount, error) {
	return s.State.InitialBalance, nil
}

func (s *state0) Threshold() (uint64, error) {
	return s.State.NumApprovalsThreshold, nil
}

func (s *state0) Signers() ([]address.Address, error) {
	return s.State.Signers, nil
}

func (s *state0) ForEachPendingTxn(cb func(id int64, txn Transaction) error) error {
	arr, err := adt0.AsMap(s.store, s.State.PendingTxns)
	if err != nil {
		return err
	}
	var out msig0.Transaction
	return arr.ForEach(&out, func(key string) error {
		txid, n := binary.Varint([]byte(key))
		if n <= 0 {
			return xerrors.Errorf("invalid pending transaction key: %v", key)
		}
		return cb(txid, (Transaction)(out)) //nolint:unconvert
	})
}

func (s *state0) PendingTxnChanged(other State) (bool, error) {
	other0, ok := other.(*state0)
	if !ok {
		// treat an upgrade as a change, always
		return true, nil
	}
	return !s.State.PendingTxns.Equals(other0.PendingTxns), nil
}

func (s *state0) transactions() (adt.Map, error) {
	return adt0.AsMap(s.store, s.PendingTxns)
}

func (s *state0) decodeTransaction(val *cbg.Deferred) (Transaction, error) {
	var tx msig0.Transaction
	if err := tx.UnmarshalCBOR(bytes.NewReader(val.Raw)); err != nil {
		return Transaction{}, err
	}
	return tx, nil
}

func (s *state0) GetState() interface{} {
	return &s.State
}
