package market

import (
	"bytes"

	"github.com/filecoin-project/go-address"
	"github.com/filecoin-project/go-state-types/abi"
	"github.com/ipfs/go-cid"
	cbg "github.com/whyrusleeping/cbor-gen"

	"github.com/filecoin-project/lotus/chain/actors/adt"
	"github.com/filecoin-project/lotus/chain/types"

	market3 "github.com/filecoin-project/specs-actors/v3/actors/builtin/market"
	adt3 "github.com/filecoin-project/specs-actors/v3/actors/util/adt"
)

var _ State = (*state3)(nil)

func load3(store adt.Store, root cid.Cid) (State, error) {
	out := state3{store: store}
	err := store.Get(store.Context(), root, &out)
	if err != nil {
		return nil, err
	}
	return &out, nil
}

func make3(store adt.Store) (State, error) {
	out := state3{store: store}

	s, err := market3.ConstructState(store)
	if err != nil {
		return nil, err
	}

	out.State = *s

	return &out, nil
}

type state3 struct {
	market3.State
	store adt.Store
}

func (s *state3) TotalLocked() (abi.TokenAmount, error) {
	fml := types.BigAdd(s.TotalClientLockedCollateral, s.TotalProviderLockedCollateral)
	fml = types.BigAdd(fml, s.TotalClientStorageFee)
	return fml, nil
}

func (s *state3) BalancesChanged(otherState State) (bool, error) {
	otherState3, ok := otherState.(*state3)
	if !ok {
		// there's no way to compare different versions of the state, so let's
		// just say that means the state of balances has changed
		return true, nil
	}
	return !s.State.EscrowTable.Equals(otherState3.State.EscrowTable) || !s.State.LockedTable.Equals(otherState3.State.LockedTable), nil
}

func (s *state3) StatesChanged(otherState State) (bool, error) {
	otherState3, ok := otherState.(*state3)
	if !ok {
		// there's no way to compare different versions of the state, so let's
		// just say that means the state of balances has changed
		return true, nil
	}
	return !s.State.States.Equals(otherState3.State.States), nil
}

func (s *state3) States() (DealStates, error) {
	stateArray, err := adt3.AsArray(s.store, s.State.States, market3.StatesAmtBitwidth)
	if err != nil {
		return nil, err
	}
	return &dealStates3{stateArray}, nil
}

func (s *state3) ProposalsChanged(otherState State) (bool, error) {
	otherState3, ok := otherState.(*state3)
	if !ok {
		// there's no way to compare different versions of the state, so let's
		// just say that means the state of balances has changed
		return true, nil
	}
	return !s.State.Proposals.Equals(otherState3.State.Proposals), nil
}

func (s *state3) Proposals() (DealProposals, error) {
	proposalArray, err := adt3.AsArray(s.store, s.State.Proposals, market3.ProposalsAmtBitwidth)
	if err != nil {
		return nil, err
	}
	return &dealProposals3{proposalArray}, nil
}

func (s *state3) EscrowTable() (BalanceTable, error) {
	bt, err := adt3.AsBalanceTable(s.store, s.State.EscrowTable)
	if err != nil {
		return nil, err
	}
	return &balanceTable3{bt}, nil
}

func (s *state3) LockedTable() (BalanceTable, error) {
	bt, err := adt3.AsBalanceTable(s.store, s.State.LockedTable)
	if err != nil {
		return nil, err
	}
	return &balanceTable3{bt}, nil
}

func (s *state3) VerifyDealsForActivation(
	minerAddr address.Address, deals []abi.DealID, currEpoch, sectorExpiry abi.ChainEpoch,
) (weight, verifiedWeight abi.DealWeight, err error) {
	w, vw, _, err := market3.ValidateDealsForActivation(&s.State, s.store, deals, minerAddr, sectorExpiry, currEpoch)
	return w, vw, err
}

func (s *state3) NextID() (abi.DealID, error) {
	return s.State.NextID, nil
}

type balanceTable3 struct {
	*adt3.BalanceTable
}

func (bt *balanceTable3) ForEach(cb func(address.Address, abi.TokenAmount) error) error {
	asMap := (*adt3.Map)(bt.BalanceTable)
	var ta abi.TokenAmount
	return asMap.ForEach(&ta, func(key string) error {
		a, err := address.NewFromBytes([]byte(key))
		if err != nil {
			return err
		}
		return cb(a, ta)
	})
}

type dealStates3 struct {
	adt.Array
}

func (s *dealStates3) Get(dealID abi.DealID) (*DealState, bool, error) {
	var deal3 market3.DealState
	found, err := s.Array.Get(uint64(dealID), &deal3)
	if err != nil {
		return nil, false, err
	}
	if !found {
		return nil, false, nil
	}
	deal := fromV3DealState(deal3)
	return &deal, true, nil
}

func (s *dealStates3) ForEach(cb func(dealID abi.DealID, ds DealState) error) error {
	var ds3 market3.DealState
	return s.Array.ForEach(&ds3, func(idx int64) error {
		return cb(abi.DealID(idx), fromV3DealState(ds3))
	})
}

func (s *dealStates3) decode(val *cbg.Deferred) (*DealState, error) {
	var ds3 market3.DealState
	if err := ds3.UnmarshalCBOR(bytes.NewReader(val.Raw)); err != nil {
		return nil, err
	}
	ds := fromV3DealState(ds3)
	return &ds, nil
}

func (s *dealStates3) array() adt.Array {
	return s.Array
}

func fromV3DealState(v3 market3.DealState) DealState {
	return (DealState)(v3)
}

type dealProposals3 struct {
	adt.Array
}

func (s *dealProposals3) Get(dealID abi.DealID) (*DealProposal, bool, error) {
	var proposal3 market3.DealProposal
	found, err := s.Array.Get(uint64(dealID), &proposal3)
	if err != nil {
		return nil, false, err
	}
	if !found {
		return nil, false, nil
	}
	proposal := fromV3DealProposal(proposal3)
	return &proposal, true, nil
}

func (s *dealProposals3) ForEach(cb func(dealID abi.DealID, dp DealProposal) error) error {
	var dp3 market3.DealProposal
	return s.Array.ForEach(&dp3, func(idx int64) error {
		return cb(abi.DealID(idx), fromV3DealProposal(dp3))
	})
}

func (s *dealProposals3) decode(val *cbg.Deferred) (*DealProposal, error) {
	var dp3 market3.DealProposal
	if err := dp3.UnmarshalCBOR(bytes.NewReader(val.Raw)); err != nil {
		return nil, err
	}
	dp := fromV3DealProposal(dp3)
	return &dp, nil
}

func (s *dealProposals3) array() adt.Array {
	return s.Array
}

func fromV3DealProposal(v3 market3.DealProposal) DealProposal {
	return (DealProposal)(v3)
}

func (s *state3) GetState() interface{} {
	return &s.State
}
