package market

import (
	"bytes"

	"github.com/filecoin-project/go-address"
	"github.com/filecoin-project/go-state-types/abi"
	"github.com/ipfs/go-cid"
	cbg "github.com/whyrusleeping/cbor-gen"

	"github.com/filecoin-project/lotus/chain/actors/adt"
	"github.com/filecoin-project/lotus/chain/types"

	market0 "github.com/filecoin-project/specs-actors/actors/builtin/market"
	adt0 "github.com/filecoin-project/specs-actors/actors/util/adt"
)

var _ State = (*state0)(nil)

func load0(store adt.Store, root cid.Cid) (State, error) {
	out := state0{store: store}
	err := store.Get(store.Context(), root, &out)
	if err != nil {
		return nil, err
	}
	return &out, nil
}

func make0(store adt.Store) (State, error) {
	out := state0{store: store}

	ea, err := adt0.MakeEmptyArray(store).Root()
	if err != nil {
		return nil, err
	}

	em, err := adt0.MakeEmptyMap(store).Root()
	if err != nil {
		return nil, err
	}

	out.State = *market0.ConstructState(ea, em, em)

	return &out, nil
}

type state0 struct {
	market0.State
	store adt.Store
}

func (s *state0) TotalLocked() (abi.TokenAmount, error) {
	fml := types.BigAdd(s.TotalClientLockedCollateral, s.TotalProviderLockedCollateral)
	fml = types.BigAdd(fml, s.TotalClientStorageFee)
	return fml, nil
}

func (s *state0) BalancesChanged(otherState State) (bool, error) {
	otherState0, ok := otherState.(*state0)
	if !ok {
		// there's no way to compare different versions of the state, so let's
		// just say that means the state of balances has changed
		return true, nil
	}
	return !s.State.EscrowTable.Equals(otherState0.State.EscrowTable) || !s.State.LockedTable.Equals(otherState0.State.LockedTable), nil
}

func (s *state0) StatesChanged(otherState State) (bool, error) {
	otherState0, ok := otherState.(*state0)
	if !ok {
		// there's no way to compare different versions of the state, so let's
		// just say that means the state of balances has changed
		return true, nil
	}
	return !s.State.States.Equals(otherState0.State.States), nil
}

func (s *state0) States() (DealStates, error) {
	stateArray, err := adt0.AsArray(s.store, s.State.States)
	if err != nil {
		return nil, err
	}
	return &dealStates0{stateArray}, nil
}

func (s *state0) ProposalsChanged(otherState State) (bool, error) {
	otherState0, ok := otherState.(*state0)
	if !ok {
		// there's no way to compare different versions of the state, so let's
		// just say that means the state of balances has changed
		return true, nil
	}
	return !s.State.Proposals.Equals(otherState0.State.Proposals), nil
}

func (s *state0) Proposals() (DealProposals, error) {
	proposalArray, err := adt0.AsArray(s.store, s.State.Proposals)
	if err != nil {
		return nil, err
	}
	return &dealProposals0{proposalArray}, nil
}

func (s *state0) EscrowTable() (BalanceTable, error) {
	bt, err := adt0.AsBalanceTable(s.store, s.State.EscrowTable)
	if err != nil {
		return nil, err
	}
	return &balanceTable0{bt}, nil
}

func (s *state0) LockedTable() (BalanceTable, error) {
	bt, err := adt0.AsBalanceTable(s.store, s.State.LockedTable)
	if err != nil {
		return nil, err
	}
	return &balanceTable0{bt}, nil
}

func (s *state0) VerifyDealsForActivation(
	minerAddr address.Address, deals []abi.DealID, currEpoch, sectorExpiry abi.ChainEpoch,
) (weight, verifiedWeight abi.DealWeight, err error) {
	w, vw, err := market0.ValidateDealsForActivation(&s.State, s.store, deals, minerAddr, sectorExpiry, currEpoch)
	return w, vw, err
}

func (s *state0) NextID() (abi.DealID, error) {
	return s.State.NextID, nil
}

type balanceTable0 struct {
	*adt0.BalanceTable
}

func (bt *balanceTable0) ForEach(cb func(address.Address, abi.TokenAmount) error) error {
	asMap := (*adt0.Map)(bt.BalanceTable)
	var ta abi.TokenAmount
	return asMap.ForEach(&ta, func(key string) error {
		a, err := address.NewFromBytes([]byte(key))
		if err != nil {
			return err
		}
		return cb(a, ta)
	})
}

type dealStates0 struct {
	adt.Array
}

func (s *dealStates0) Get(dealID abi.DealID) (*DealState, bool, error) {
	var deal0 market0.DealState
	found, err := s.Array.Get(uint64(dealID), &deal0)
	if err != nil {
		return nil, false, err
	}
	if !found {
		return nil, false, nil
	}
	deal := fromV0DealState(deal0)
	return &deal, true, nil
}

func (s *dealStates0) ForEach(cb func(dealID abi.DealID, ds DealState) error) error {
	var ds0 market0.DealState
	return s.Array.ForEach(&ds0, func(idx int64) error {
		return cb(abi.DealID(idx), fromV0DealState(ds0))
	})
}

func (s *dealStates0) decode(val *cbg.Deferred) (*DealState, error) {
	var ds0 market0.DealState
	if err := ds0.UnmarshalCBOR(bytes.NewReader(val.Raw)); err != nil {
		return nil, err
	}
	ds := fromV0DealState(ds0)
	return &ds, nil
}

func (s *dealStates0) array() adt.Array {
	return s.Array
}

func fromV0DealState(v0 market0.DealState) DealState {
	return (DealState)(v0)
}

type dealProposals0 struct {
	adt.Array
}

func (s *dealProposals0) Get(dealID abi.DealID) (*DealProposal, bool, error) {
	var proposal0 market0.DealProposal
	found, err := s.Array.Get(uint64(dealID), &proposal0)
	if err != nil {
		return nil, false, err
	}
	if !found {
		return nil, false, nil
	}
	proposal := fromV0DealProposal(proposal0)
	return &proposal, true, nil
}

func (s *dealProposals0) ForEach(cb func(dealID abi.DealID, dp DealProposal) error) error {
	var dp0 market0.DealProposal
	return s.Array.ForEach(&dp0, func(idx int64) error {
		return cb(abi.DealID(idx), fromV0DealProposal(dp0))
	})
}

func (s *dealProposals0) decode(val *cbg.Deferred) (*DealProposal, error) {
	var dp0 market0.DealProposal
	if err := dp0.UnmarshalCBOR(bytes.NewReader(val.Raw)); err != nil {
		return nil, err
	}
	dp := fromV0DealProposal(dp0)
	return &dp, nil
}

func (s *dealProposals0) array() adt.Array {
	return s.Array
}

func fromV0DealProposal(v0 market0.DealProposal) DealProposal {
	return (DealProposal)(v0)
}

func (s *state0) GetState() interface{} {
	return &s.State
}
