package paych

import (
	"github.com/ipfs/go-cid"

	"github.com/filecoin-project/go-address"
	"github.com/filecoin-project/go-state-types/abi"
	"github.com/filecoin-project/go-state-types/big"

	"github.com/filecoin-project/lotus/chain/actors/adt"

	paych3 "github.com/filecoin-project/specs-actors/v3/actors/builtin/paych"
	adt3 "github.com/filecoin-project/specs-actors/v3/actors/util/adt"
)

var _ State = (*state3)(nil)

func load3(store adt.Store, root cid.Cid) (State, error) {
	out := state3{store: store}
	err := store.Get(store.Context(), root, &out)
	if err != nil {
		return nil, err
	}
	return &out, nil
}

func make3(store adt.Store) (State, error) {
	out := state3{store: store}
	out.State = paych3.State{}
	return &out, nil
}

type state3 struct {
	paych3.State
	store adt.Store
	lsAmt *adt3.Array
}

// Channel owner, who has funded the actor
func (s *state3) From() (address.Address, error) {
	return s.State.From, nil
}

// Recipient of payouts from channel
func (s *state3) To() (address.Address, error) {
	return s.State.To, nil
}

// Height at which the channel can be `Collected`
func (s *state3) SettlingAt() (abi.ChainEpoch, error) {
	return s.State.SettlingAt, nil
}

// Amount successfully redeemed through the payment channel, paid out on `Collect()`
func (s *state3) ToSend() (abi.TokenAmount, error) {
	return s.State.ToSend, nil
}

func (s *state3) getOrLoadLsAmt() (*adt3.Array, error) {
	if s.lsAmt != nil {
		return s.lsAmt, nil
	}

	// Get the lane state from the chain
	lsamt, err := adt3.AsArray(s.store, s.State.LaneStates, paych3.LaneStatesAmtBitwidth)
	if err != nil {
		return nil, err
	}

	s.lsAmt = lsamt
	return lsamt, nil
}

// Get total number of lanes
func (s *state3) LaneCount() (uint64, error) {
	lsamt, err := s.getOrLoadLsAmt()
	if err != nil {
		return 0, err
	}
	return lsamt.Length(), nil
}

func (s *state3) GetState() interface{} {
	return &s.State
}

// Iterate lane states
func (s *state3) ForEachLaneState(cb func(idx uint64, dl LaneState) error) error {
	// Get the lane state from the chain
	lsamt, err := s.getOrLoadLsAmt()
	if err != nil {
		return err
	}

	// Note: we use a map instead of an array to store laneStates because the
	// client sets the lane ID (the index) and potentially they could use a
	// very large index.
	var ls paych3.LaneState
	return lsamt.ForEach(&ls, func(i int64) error {
		return cb(uint64(i), &laneState3{ls})
	})
}

type laneState3 struct {
	paych3.LaneState
}

func (ls *laneState3) Redeemed() (big.Int, error) {
	return ls.LaneState.Redeemed, nil
}

func (ls *laneState3) Nonce() (uint64, error) {
	return ls.LaneState.Nonce, nil
}
