package init

import (
	"github.com/filecoin-project/go-address"
	"github.com/filecoin-project/go-state-types/abi"
	"github.com/ipfs/go-cid"
	cbg "github.com/whyrusleeping/cbor-gen"
	"golang.org/x/xerrors"

	"github.com/filecoin-project/lotus/chain/actors/adt"
	"github.com/filecoin-project/lotus/node/modules/dtypes"

	init0 "github.com/filecoin-project/specs-actors/actors/builtin/init"
	adt0 "github.com/filecoin-project/specs-actors/actors/util/adt"
)

var _ State = (*state0)(nil)

func load0(store adt.Store, root cid.Cid) (State, error) {
	out := state0{store: store}
	err := store.Get(store.Context(), root, &out)
	if err != nil {
		return nil, err
	}
	return &out, nil
}

func make0(store adt.Store, networkName string) (State, error) {
	out := state0{store: store}

	mr, err := adt0.MakeEmptyMap(store).Root()
	if err != nil {
		return nil, err
	}

	out.State = *init0.ConstructState(mr, networkName)

	return &out, nil
}

type state0 struct {
	init0.State
	store adt.Store
}

func (s *state0) ResolveAddress(address address.Address) (address.Address, bool, error) {
	return s.State.ResolveAddress(s.store, address)
}

func (s *state0) MapAddressToNewID(address address.Address) (address.Address, error) {
	return s.State.MapAddressToNewID(s.store, address)
}

func (s *state0) ForEachActor(cb func(id abi.ActorID, address address.Address) error) error {
	addrs, err := adt0.AsMap(s.store, s.State.AddressMap)
	if err != nil {
		return err
	}
	var actorID cbg.CborInt
	return addrs.ForEach(&actorID, func(key string) error {
		addr, err := address.NewFromBytes([]byte(key))
		if err != nil {
			return err
		}
		return cb(abi.ActorID(actorID), addr)
	})
}

func (s *state0) NetworkName() (dtypes.NetworkName, error) {
	return dtypes.NetworkName(s.State.NetworkName), nil
}

func (s *state0) SetNetworkName(name string) error {
	s.State.NetworkName = name
	return nil
}

func (s *state0) SetNextID(id abi.ActorID) error {
	s.State.NextID = id
	return nil
}

func (s *state0) Remove(addrs ...address.Address) (err error) {
	m, err := adt0.AsMap(s.store, s.State.AddressMap)
	if err != nil {
		return err
	}
	for _, addr := range addrs {
		if err = m.Delete(abi.AddrKey(addr)); err != nil {
			return xerrors.Errorf("failed to delete entry for address: %s; err: %w", addr, err)
		}
	}
	amr, err := m.Root()
	if err != nil {
		return xerrors.Errorf("failed to get address map root: %w", err)
	}
	s.State.AddressMap = amr
	return nil
}

func (s *state0) SetAddressMap(mcid cid.Cid) error {
	s.State.AddressMap = mcid
	return nil
}

func (s *state0) AddressMap() (adt.Map, error) {
	return adt0.AsMap(s.store, s.State.AddressMap)
}

func (s *state0) GetState() interface{} {
	return &s.State
}
