package seal

import (
	"bytes"
	"context"
	"fmt"
	"io"
	"os"
	"io/ioutil"
	"runtime"
	"sync"

	"golang.org/x/xerrors"

	ffi "github.com/a263200357/filecoin-ffi"
	logging "github.com/ipfs/go-log/v2"

	"fil_integrate/build/cid"
	"fil_integrate/build/fr32"
	spproof "fil_integrate/build/proof"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/build/storiface"
)

var log = logging.Logger("sealing")

var PicesNotEnoughError = xerrors.Errorf("can not use the existing pieces to fill the sector")

type Sealer struct {
	sectors      SectorManager
	sortedPieces []abi.PieceInfo
}

var _ SectorSealer = &Sealer{}

func NewSealer(sectors SectorManager) *Sealer {
	sb := &Sealer{
		sectors: sectors,
	}

	return sb
}

func (sb *Sealer) SavePiece(ctx context.Context, piece abi.PieceInfo, in storage.Data) error {
	var res []abi.PieceInfo

	if in != nil {
		stagePath, done, err := sb.sectors.AcquirePiece(ctx, piece.PieceCID, 0, storiface.FTPiece)
		if err != nil {
			return err
		}
		defer done()
		out, err := os.OpenFile(stagePath.Piece, os.O_WRONLY|os.O_CREATE, 0644)
		if err != nil {
			return err
		}
		_, err = io.CopyN(out, in, int64(piece.Size.Unpadded()))
		if err != nil {
			return err
		}
	}

	if sb.sortedPieces == nil || piece.Size >= sb.sortedPieces[0].Size {
		res = append(res, piece)
		res = append(res, sb.sortedPieces...)
	} else {
		var i int
		for i = len(sb.sortedPieces) - 1; i >= 0; i-- {
			if sb.sortedPieces[i].Size >= piece.Size {
				break
			}
		}
		res = append(res, sb.sortedPieces[:i+1]...)
		res = append(res, piece)
		res = append(res, sb.sortedPieces[i+1:]...)
	}
	sb.sortedPieces = res
	return nil
}

func (sb *Sealer) AddPiece(
	ctx context.Context,
	sector storage.SectorRef,
) ([]abi.PieceInfo, error) {
	var index int
	var addPieces []abi.PieceInfo
	var pieceSize abi.PaddedPieceSize
	var existingPieceSizes []abi.UnpaddedPieceSize
	var piecesInfo []abi.PieceInfo

	ssize, err := sector.ProofType.SectorSize()
	if err != nil {
		return nil, err
	}

	maxPieceSize := abi.PaddedPieceSize(ssize)

	// Select pieces to seal
	for index = 0; index < len(sb.sortedPieces); index++ {
		if pieceSize > maxPieceSize {
			return nil, xerrors.Errorf("Exists a piece whose size is bigger than 8MiB or is not power of two or the pieces is not sorted")
		} else if pieceSize == maxPieceSize {
			break
		}
		pieceSize += sb.sortedPieces[index].Size
		addPieces = append(addPieces, sb.sortedPieces[index])
	}

	if pieceSize != maxPieceSize {
		return nil, PicesNotEnoughError
	}

	// Seal the selected pieces
	for _, piece := range addPieces {
		stagePath, done, err := sb.sectors.AcquirePiece(ctx, piece.PieceCID, storiface.FTPiece, 0)
		if err != nil {
			return nil, err
		}
		file, err := os.Open(stagePath.Piece)
		if err != nil {
			return nil, err
		}
		defer func() {
			done()
			file.Close()
			os.Remove(stagePath.Piece)
		}()
		fmt.Printf("Adding %s\n", stagePath.Piece)
		pieceInfo, err := sb.addPiece(ctx, sector, existingPieceSizes, piece.Size.Unpadded(), file)
		if err != nil {
			return nil, err
		}
		existingPieceSizes = append(existingPieceSizes, piece.Size.Unpadded())
		piecesInfo = append(piecesInfo, pieceInfo)
	}
	sb.sortedPieces = sb.sortedPieces[index:]

	return piecesInfo, nil
}

func (sb *Sealer) addPiece(
	ctx context.Context,
	sector storage.SectorRef,
	existingPieceSizes []abi.UnpaddedPieceSize,
	pieceSize abi.UnpaddedPieceSize,
	file storage.Data,
) (abi.PieceInfo, error) {
	chunk := abi.PaddedPieceSize(4 << 20)
	parallel := runtime.NumCPU()

	var offset abi.UnpaddedPieceSize
	for _, size := range existingPieceSizes {
		offset += size
	}

	ssize, err := sector.ProofType.SectorSize()
	if err != nil {
		return abi.PieceInfo{}, err
	}

	maxPieceSize := abi.PaddedPieceSize(ssize)

	if offset.Padded()+pieceSize.Padded() > maxPieceSize {
		return abi.PieceInfo{}, xerrors.Errorf("can't add %d byte piece to sector %v with %d bytes of existing pieces with %d bytes of max bytes", pieceSize, sector, offset, maxPieceSize)
	}

	var done func()
	var stagedFile *os.File

	defer func() {
		if done != nil {
			done()
		}

		if stagedFile != nil {
			if err := stagedFile.Close(); err != nil {
				log.Errorf("closing staged file: %+v", err)
			}
		}
	}()

	var stagedPath storiface.SectorPaths
	if len(existingPieceSizes) == 0 {
		stagedPath, done, err = sb.sectors.AcquireSector(ctx, sector, 0, storiface.FTUnsealed)
		if err != nil {
			return abi.PieceInfo{}, xerrors.Errorf("acquire unsealed sector: %w", err)
		}

		stagedFile, err = os.OpenFile(stagedPath.Unsealed, os.O_RDWR|os.O_CREATE, 0644) // nolint
		if err != nil {
			return abi.PieceInfo{}, xerrors.Errorf("openning file '%s': %w", stagedFile, err)
		}

	} else {
		stagedPath, done, err = sb.sectors.AcquireSector(ctx, sector, storiface.FTUnsealed, 0)
		if err != nil {
			return abi.PieceInfo{}, xerrors.Errorf("acquire unsealed sector: %w", err)
		}

		stagedFile, err = os.OpenFile(stagedPath.Unsealed, os.O_RDWR|os.O_APPEND, 0644) // nolint
		if err != nil {
			return abi.PieceInfo{}, xerrors.Errorf("openning file '%s': %w", stagedFile, err)
		}
	}

	pw := fr32.NewPadWriter(stagedFile)

	pr := io.TeeReader(io.LimitReader(file, int64(pieceSize)), pw)

	throttle := make(chan []byte, parallel)
	piecePromises := make([]func() (abi.PieceInfo, error), 0)

	buf := make([]byte, chunk.Unpadded())
	for i := 0; i < parallel; i++ {
		if abi.UnpaddedPieceSize(i)*chunk.Unpadded() >= pieceSize {
			break // won't use this many buffers
		}
		throttle <- make([]byte, chunk.Unpadded())
	}

	for {
		var read int
		for rbuf := buf; len(rbuf) > 0; {
			n, err := pr.Read(rbuf)
			if err != nil && err != io.EOF {
				return abi.PieceInfo{}, xerrors.Errorf("pr read error: %w", err)
			}

			rbuf = rbuf[n:]
			read += n

			if err == io.EOF {
				break
			}
		}
		if read == 0 {
			break
		}

		done := make(chan struct {
			cid.Commit
			error
		}, 1)
		pbuf := <-throttle
		copy(pbuf, buf[:read])

		go func(read int) {
			defer func() {
				throttle <- pbuf
			}()

			c, err := pieceCid(sector.ProofType, pbuf[:read])
			done <- struct {
				cid.Commit
				error
			}{c, err}
		}(read)

		piecePromises = append(piecePromises, func() (abi.PieceInfo, error) {
			select {
			case e := <-done:
				if e.error != nil {
					return abi.PieceInfo{}, e.error
				}

				return abi.PieceInfo{
					Size:     abi.UnpaddedPieceSize(len(buf[:read])).Padded(),
					PieceCID: e.Commit,
				}, nil
			case <-ctx.Done():
				return abi.PieceInfo{}, ctx.Err()
			}
		})
	}

	if err := pw.Close(); err != nil {
		return abi.PieceInfo{}, xerrors.Errorf("closing padded writer: %w", err)
	}

	if err := stagedFile.Close(); err != nil {
		return abi.PieceInfo{}, err
	}
	stagedFile = nil

	if len(piecePromises) == 1 {
		return piecePromises[0]()
	}

	pieceCids := make([]abi.PieceInfo, len(piecePromises))
	for i, promise := range piecePromises {
		pieceCids[i], err = promise()
		if err != nil {
			return abi.PieceInfo{}, err
		}
	}

	pieceCID, err := ffi.GenerateUnsealedCID(sector.ProofType, pieceCids)
	if err != nil {
		return abi.PieceInfo{}, xerrors.Errorf("generate unsealed CID: %w", err)
	}

	return abi.PieceInfo{
		Size:     pieceSize.Padded(),
		PieceCID: pieceCID,
	}, nil

}

func pieceCid(spt abi.RegisteredSealProof, in []byte) (cid.Commit, error) {
	prf, werr, err := toReadableFile(bytes.NewReader(in), int64(len(in)))
	if err != nil {
		return cid.Undef, xerrors.Errorf("getting tee reader pipe: %w", err)
	}

	pieceCID, err := ffi.GeneratePieceCIDFromFile(spt, prf, abi.UnpaddedPieceSize(len(in)))
	if err != nil {
		return cid.Undef, xerrors.Errorf("generating piece commitment: %w", err)
	}

	_ = prf.Close()

	return pieceCID, werr()
}

func toReadableFile(r io.Reader, n int64) (*os.File, func() error, error) {
	f, ok := r.(*os.File)
	if ok {
		return f, func() error { return nil }, nil
	}

	var w *os.File

	f, w, err := os.Pipe()
	if err != nil {
		return nil, nil, err
	}

	var wait sync.Mutex
	var werr error

	wait.Lock()
	go func() {
		defer wait.Unlock()

		var copied int64
		copied, werr = io.CopyN(w, r, n)
		if werr != nil {
			log.Warnf("toReadableFile: copy error: %+v", werr)
		}

		err := w.Close()
		if werr == nil && err != nil {
			werr = err
			log.Warnf("toReadableFile: close error: %+v", err)
			return
		}
		if copied != n {
			log.Warnf("copied different amount than expected: %d != %d", copied, n)
			werr = xerrors.Errorf("copied different amount than expected: %d != %d", copied, n)
		}
	}()

	return f, func() error {
		wait.Lock()
		return werr
	}, nil
}

func (sb *Sealer) ReadPiece(ctx context.Context, pieceID cid.Commit) ([]byte, error) {
	srcPaths, srcDone, err := sb.sectors.AcquirePiece(ctx, pieceID, storiface.FTPiece, storiface.FTNone)
	if err != nil {
		return nil, xerrors.Errorf("acquire sealed sector paths: %w", err)
	}
	defer srcDone()

	return ioutil.ReadFile(srcPaths.Piece)
}

func (sb *Sealer) UnsealedPieceRange(
	ctx context.Context,
	sid storage.SectorRef,
	unsealed cid.Commit,
	offset abi.UnpaddedByteIndex,
	size abi.UnpaddedPieceSize,
) ([]byte, error) {
	log.Infof("[%d] Unsealing sector", sid.ID.Number)

	srcPaths, srcDone, err := sb.sectors.AcquireSector(ctx, sid, storiface.FTCache|storiface.FTSealed, storiface.FTNone)
	if err != nil {
		return nil, xerrors.Errorf("acquire sealed sector paths: %w", err)
	}
	defer srcDone()

	sealed, err := os.OpenFile(srcPaths.Sealed, os.O_RDONLY, 0644) // nolint:gosec
	if err != nil {
		return nil, xerrors.Errorf("opening sealed file: %w", err)
	}
	defer sealed.Close() // nolint

	// <eww>
	opr, opw, err := os.Pipe()
	if err != nil {
		return nil, xerrors.Errorf("creating out pipe: %w", err)
	}

	var perr error
	outWait := make(chan struct{})
	buf := make([]byte, size)

	{
		go func() {
			defer close(outWait)
			defer opr.Close() // nolint

			for wbuf := buf[:]; len(wbuf) > 0; {
				n, err := opr.Read(wbuf)
				if err != nil && err != io.EOF {
					perr = xerrors.Errorf("copying data: %w", err)
					return
				}
				wbuf = wbuf[n:]
			}
		}()
	}
	// </eww>

	// TODO: This may be possible to do in parallel
	err = ffi.UnsealRange(sid.ProofType,
		srcPaths.Cache,
		sealed,
		opw,
		sid.ID.Number,
		sid.ID.Miner,
		Ticket,
		unsealed,
		uint64(offset),
		uint64(size),
	)

	_ = opw.Close()

	if err != nil {
		return nil, xerrors.Errorf("unseal range: %w", err)
	}

	select {
	case <-outWait:
	case <-ctx.Done():
		return nil, ctx.Err()
	}

	if perr != nil {
		return nil, xerrors.Errorf("piping output to unsealed file: %w", perr)
	}

	return buf, nil

}

func (sb *Sealer) SealPreCommit1(ctx context.Context, sector storage.SectorRef, pieces []abi.PieceInfo) (out storage.PreCommit1Out, err error) {
	// ffi.say_hello()
	paths, done, err := sb.sectors.AcquireSector(ctx, sector, storiface.FTUnsealed, storiface.FTSealed|storiface.FTCache)
	if err != nil {
		return nil, xerrors.Errorf("acquiring sector paths: %w", err)
	}
	defer func() {
		done()
		os.Remove(paths.Unsealed)
	}()

	e, err := os.OpenFile(paths.Sealed, os.O_RDWR|os.O_CREATE, 0644) // nolint:gosec
	if err != nil {
		return nil, xerrors.Errorf("ensuring sealed file exists: %w", err)
	}
	if err := e.Close(); err != nil {
		return nil, err
	}

	if err := os.Mkdir(paths.Cache, 0755); err != nil { // nolint
		if os.IsExist(err) {
			log.Warnf("existing cache in %s; removing", paths.Cache)

			if err := os.RemoveAll(paths.Cache); err != nil {
				return nil, xerrors.Errorf("remove existing sector cache from %s (sector %d): %w", paths.Cache, sector, err)
			}

			if err := os.Mkdir(paths.Cache, 0755); err != nil { // nolint:gosec
				return nil, xerrors.Errorf("mkdir cache path after cleanup: %w", err)
			}
		} else {
			return nil, err
		}
	}

	var sum abi.UnpaddedPieceSize
	for _, piece := range pieces {
		sum += piece.Size.Unpadded()
	}
	ssize, err := sector.ProofType.SectorSize()
	if err != nil {
		return nil, err
	}
	ussize := abi.PaddedPieceSize(ssize).Unpadded()
	if sum != ussize {
		return nil, xerrors.Errorf("aggregated piece sizes don't match sector size: %d != %d (%d)", sum, ussize, int64(ussize-sum))
	}

	// TODO: context cancellation respect
	p1o, err := ffi.SealPreCommitPhase1(
		sector.ProofType,
		paths.Cache,
		paths.Unsealed,
		paths.Sealed,
		sector.ID.Number,
		sector.ID.Miner,
		Ticket,
		pieces,
	)
	if err != nil {
		return nil, xerrors.Errorf("presealing sector %d (%s): %w", sector.ID.Number, paths.Unsealed, err)
	}
	return p1o, nil
}

func (sb *Sealer) SealPreCommit2(ctx context.Context, sector storage.SectorRef, phase1Out storage.PreCommit1Out) (storage.SectorCids, error) {
	paths, done, err := sb.sectors.AcquireSector(ctx, sector, storiface.FTSealed|storiface.FTCache, 0)
	if err != nil {
		return storage.SectorCids{}, xerrors.Errorf("acquiring sector paths: %w", err)
	}
	defer done()

	sealedCID, unsealedCID, err := ffi.SealPreCommitPhase2(phase1Out, paths.Cache, paths.Sealed)
	if err != nil {
		return storage.SectorCids{}, xerrors.Errorf("presealing sector %d (%s): %w", sector.ID.Number, paths.Unsealed, err)
	}

	return storage.SectorCids{
		Unsealed: unsealedCID,
		Sealed:   sealedCID,
	}, nil
}

func (sb *Sealer) SealCommit1(ctx context.Context, sector storage.SectorRef, seed abi.InteractiveSealRandomness, pieces []abi.PieceInfo, cids storage.SectorCids) (storage.Commit1Out, error) {
	paths, done, err := sb.sectors.AcquireSector(ctx, sector, storiface.FTSealed|storiface.FTCache, 0)
	if err != nil {
		return nil, xerrors.Errorf("acquire sector paths: %w", err)
	}
	defer done()
	output, err := ffi.SealCommitPhase1(
		sector.ProofType,
		cids.Sealed,
		cids.Unsealed,
		paths.Cache,
		paths.Sealed,
		sector.ID.Number,
		sector.ID.Miner,
		Ticket,
		seed,
		pieces,
	)
	if err != nil {
		log.Warn("StandaloneSealCommit error: ", err)
		log.Warnf("num:%d tkt:%v seed:%v, pi:%v sealedCID:%v, unsealedCID:%v", sector.ID.Number, Ticket, seed, pieces, cids.Sealed, cids.Unsealed)

		return nil, xerrors.Errorf("StandaloneSealCommit: %w", err)
	}
	return output, nil
}

func (sb *Sealer) SealCommit2(ctx context.Context, sector storage.SectorRef, phase1Out storage.Commit1Out) ([]byte, error) {
	return ffi.SealCommitPhase2(phase1Out, sector.ID.Number, sector.ID.Miner)
}

func (sb *Sealer) Sealed(
	ctx context.Context,
	sid storage.SectorRef,
	pieces []abi.PieceInfo,
) (storage.SectorCids, error) {
	// var sealedSectors spproof.SectorInfo

	log.Infof("[%d] Running replication(1)...", sid.ID.Number)

	pc1out, err := sb.SealPreCommit1(ctx, sid, pieces)
	if err != nil {
		return storage.SectorCids{}, xerrors.Errorf("commit: %w", err)
	}

	log.Infof("[%d] Running replication(2)...", sid.ID.Number)
	cids, err := sb.SealPreCommit2(ctx, sid, pc1out)
	if err != nil {
		return storage.SectorCids{}, xerrors.Errorf("commit: %w", err)
	}

	return cids, nil
}

func (sb *Sealer) GenerateCommitProof(
	ctx context.Context,
	sid storage.SectorRef,
	seed abi.InteractiveSealRandomness,
	pieces []abi.PieceInfo,
	cids storage.SectorCids,
) (spproof.Proof, error) {

	log.Infof("[%d] Generating PoRep for sector (1)", sid.ID.Number)
	c1out, err := sb.SealCommit1(ctx, sid, seed, pieces, cids)
	if err != nil {
		return nil, err
	}

	log.Infof("[%d] Generating PoRep for sector (2)", sid.ID.Number)
	return sb.SealCommit2(ctx, sid, c1out)
}

func (sb *Sealer) AggregateSealProofs(aggregateInfo spproof.AggregateSealVerifyProofAndInfos, proofs []spproof.Proof) (spproof.Proof, error) {
	return ffi.AggregateSealProofs(aggregateInfo, proofs)
}

func (sb *Sealer) GenerateWindowPoStProofs(
	ctx context.Context,
	minerID abi.ActorID,
	sectorInfo []spproof.SectorInfo,
	randomness abi.PoStRandomness,
) (spproof.PoStProof, []abi.SectorID, error) {
	randomness[31] &= 0x3f
	privsectors, skipped, done, err := sb.pubSectorToPriv(ctx, minerID, sectorInfo, nil, abi.RegisteredSealProof.RegisteredWindowPoStProof)
	if err != nil {
		return spproof.PoStProof{}, nil, xerrors.Errorf("gathering sector info: %w", err)
	}
	defer done()

	if len(skipped) > 0 {
		return spproof.PoStProof{}, skipped, xerrors.Errorf("pubSectorToPriv skipped some sectors")
	}

	log.Infof("Generating Window-PoSt Proof")
	proof, faulty, err := ffi.GenerateWindowPoSt(minerID, privsectors, randomness)

	var faultyIDs []abi.SectorID
	for _, f := range faulty {
		faultyIDs = append(faultyIDs, abi.SectorID{
			Miner:  minerID,
			Number: f,
		})
	}

	return proof, faultyIDs, err
}

func (sb *Sealer) AggregateWindowPoStProofs(aggregateInfo spproof.AggregateWindowPostInfos, proofs []spproof.PoStProof) (spproof.PoStProof, error) {
	if len(proofs) != len(aggregateInfo.SectorCount) {
		return spproof.PoStProof{}, xerrors.Errorf("the lenth of windowPoStProofs and sectorCount is not match")
	}
	if len(aggregateInfo.Randomnesses) != len(aggregateInfo.SectorCount) {
		return spproof.PoStProof{}, xerrors.Errorf("the lenth of windowPoStProofs and randomness is not match")
	}

	sectorCount := aggregateInfo.SectorCount[0]
	for _, count := range aggregateInfo.SectorCount {
		if sectorCount != count {
			return spproof.PoStProof{}, xerrors.Errorf("Window PoSt challenge count must be equal")
		}
	}

	for i, random := range aggregateInfo.Randomnesses {
		aggregateInfo.Randomnesses[i][31] = random[31] & 0x3f
	}
	return ffi.AggregateWindowPoStProofs(aggregateInfo, proofs)
}

func (sb *Sealer) pubSectorToPriv(ctx context.Context, mid abi.ActorID, sectorInfo []spproof.SectorInfo, faults []abi.SectorNumber, rpt func(abi.RegisteredSealProof) (abi.RegisteredPoStProof, error)) (ffi.SortedPrivateSectorInfo, []abi.SectorID, func(), error) {
	fmap := map[abi.SectorNumber]struct{}{}
	for _, fault := range faults {
		fmap[fault] = struct{}{}
	}

	var doneFuncs []func()
	done := func() {
		for _, df := range doneFuncs {
			df()
		}
	}

	var skipped []abi.SectorID
	var out []ffi.PrivateSectorInfo
	for _, s := range sectorInfo {
		if _, faulty := fmap[s.SectorNumber]; faulty {
			continue
		}

		sid := storage.SectorRef{
			ID:        abi.SectorID{Miner: mid, Number: s.SectorNumber},
			ProofType: s.SealType,
		}

		paths, d, err := sb.sectors.AcquireSector(ctx, sid, storiface.FTCache|storiface.FTSealed, 0)
		if err != nil {
			log.Warnw("failed to acquire sector, skipping", "sector", sid.ID, "error", err)
			skipped = append(skipped, sid.ID)
			continue
		}
		doneFuncs = append(doneFuncs, d)

		postProofType, err := rpt(s.SealType)
		if err != nil {
			done()
			return ffi.SortedPrivateSectorInfo{}, nil, nil, xerrors.Errorf("acquiring registered PoSt proof from sector info %+v: %w", s, err)
		}

		out = append(out, ffi.PrivateSectorInfo{
			CacheDirPath:     paths.Cache,
			PoStProofType:    postProofType,
			SealedSectorPath: paths.Sealed,
			SectorInfo:       s,
		})
	}

	return ffi.NewSortedPrivateSectorInfo(out...), skipped, done, nil
}

func readFile(file string, size abi.UnpaddedPieceSize) ([]byte, error) {
	f, err := os.Open(file)
	if err != nil {
		return nil, err
	}
	defer f.Close()
	pr, err := fr32.NewUnpadReader(f, size.Padded())
	if err != nil {
		return nil, err
	}
	buf := make([]byte, size)
	for wbuf := buf[:]; len(wbuf) > 0; {
		read, err := pr.Read(wbuf)
		if err != nil {
			return nil, err
		}
		wbuf = wbuf[read:]
	}
	return buf, nil
}
