package seal

import (
	"context"
	"io"
	"io/ioutil"

	"fil_integrate/build/state-types/abi"

	"fil_integrate/build/cid"
	spieces "fil_integrate/build/pieces"
	"fil_integrate/build/storage"
	"fil_integrate/build/storiface"
	"fil_integrate/seal/basicpiece"
)

//32字节，总共256位
//[has_pre][MetaLen1..MetaLen4][PieceLen1..PieceLen4]
const TagLen uint32 = 8

type Encoder struct {
	sectors SectorManager
}

var _ PieceEncoder = &Encoder{}

func NewEncoder(sectors SectorManager) *Encoder {
	sp := &Encoder{
		sectors: sectors,
	}

	return sp
}

// Data contains MetaData and CommitData
// Pieces structure is [ Tag | MetaData | CommitData ]
func (sp *Encoder) EncodeDataToPieces(
	ctx context.Context,
	sectorSize abi.SectorSize,
	file storage.Data,
) (abi.PieceInfo, []abi.PieceInfo, error) {
	var hashData []byte
	var pieces []abi.PieceInfo
	var prePieces []abi.PieceInfo

	// root := filepath.Join(sp.Root, "pieces")

	UnpaddedSectorSize := abi.PaddedPieceSize(sectorSize).Unpadded()
	DataLen := (uint32)(UnpaddedSectorSize) - TagLen
	buf := make([]byte, DataLen)

	for {
		MetaLen, err := file.Read(buf[:])
		if err != nil && err != io.EOF {
			return abi.PieceInfo{}, nil, err
		}

		if err == io.EOF || uint32(MetaLen) != DataLen {
			//encode first sector
			prePieces, err = sp.EncodeData(ctx, buf[:uint32(MetaLen)], sectorSize, uint32(MetaLen), DataLen, hashData)
			if err != nil {
				return abi.PieceInfo{}, nil, err
			}
			break
		}

		var data *basicpiece.DecodedData = &basicpiece.DecodedData{HasPre: false, Data: buf[:]}
		dbuf, err := data.Serialize()
		if err != nil {
			return abi.PieceInfo{}, nil, err
		}

		pieceCommit, err := spieces.GeneratePieceCommitmentFast(dbuf[:], uint64(len(dbuf)))
		if err != nil {
			return abi.PieceInfo{}, nil, err
		}

		// filename := filepath.Join(root, fmt.Sprintf("%x.dat", pieceCommit[:]))
		stagePath, done, err := sp.sectors.AcquirePiece(ctx, pieceCommit, 0, storiface.FTPiece)
		if err != nil {
			return abi.PieceInfo{}, nil, err
		}
		defer done()
		err = ioutil.WriteFile(stagePath.Piece, dbuf[:], 0644)
		if err != nil {
			return abi.PieceInfo{}, nil, err
		}
		// fmt.Printf("encode1: %x.dat\n", pieceCommit[:])

		hashData = append(hashData, pieceCommit[:]...)
		pieces = append(pieces, abi.PieceInfo{
			PieceCID: pieceCommit,
			Size:     abi.PaddedPieceSize(sectorSize),
		})
	}
	pieces = append(pieces, prePieces...)
	return pieces[len(pieces)-1], pieces[:], nil
}

func (sp *Encoder) EncodeData(
	ctx context.Context,
	metadata []byte,
	sectorSize abi.SectorSize,
	MetaLen uint32,
	DataLen uint32,
	hashData []byte,
) ([]abi.PieceInfo, error) {
	// root := filepath.Join(sp.Root, "pieces")
	var prePieceCommit cid.Commit
	var pieces []abi.PieceInfo
	var err error

	if len(metadata) == 0 && len(hashData) == 32 {
		return nil, nil
	}

	for len(hashData) > 0 {
		var buf []byte
		//encode next n sector
		if pieces != nil {
			CommLen := min(uint32(len(hashData)), ((DataLen-32)/32)*32)
			var data *basicpiece.DecodedData = &basicpiece.DecodedData{
				HasPre:         true,
				PrePieceCommit: prePieceCommit,
				CommitData:     hashData[:CommLen],
			}
			buf, err = data.Serialize()
			if err != nil {
				return nil, err
			}

			hashData = hashData[CommLen:]
		} else {
			CommLen := min(uint32(len(hashData)), ((DataLen-MetaLen)/32)*32)
			var data *basicpiece.DecodedData = &basicpiece.DecodedData{
				HasPre:     false,
				Data:       metadata,
				CommitData: hashData[:CommLen],
			}
			buf, err = data.Serialize()
			if err != nil {
				return nil, err
			}

			hashData = hashData[CommLen:]
		}

		prePieceCommit, err = spieces.GeneratePieceCommitmentFast(buf, uint64(len(buf)))
		if err != nil {
			return nil, err
		}

		// filename := filepath.Join(root, fmt.Sprintf("%x.dat", prePieceCommit[:]))
		stagePath, done, err := sp.sectors.AcquirePiece(ctx, prePieceCommit, 0, storiface.FTPiece)
		if err != nil {
			return nil, err
		}
		defer done()
		err = ioutil.WriteFile(stagePath.Piece, buf, 0644)
		if err != nil {
			return nil, err
		}
		// fmt.Printf("encode2: %x.dat\n", prePieceCommit[:])

		pieces = append(pieces, abi.PieceInfo{
			PieceCID: prePieceCommit,
			Size:     abi.UnpaddedPieceSize(len(buf)).Padded(),
		})
	}

	return pieces, nil
}

func (sp *Encoder) LoadPiece(ctx context.Context, pieceID cid.Commit) ([]byte, error) {
	stagePath, done, err := sp.sectors.AcquirePiece(ctx, pieceID, storiface.FTPiece, 0)
	if err != nil {
		return nil, err
	}
	defer done()
	return ioutil.ReadFile(stagePath.Piece)
}

func (sp *Encoder) DecodePiece(
	ctx context.Context,
	buf []byte,
) (*basicpiece.DecodedData, error) {
	var data *basicpiece.DecodedData = &basicpiece.DecodedData{}
	err := data.Deserialize(buf[:])
	return data, err
}

func min(x, y uint32) uint32 {
	if x < y {
		return x
	}
	return y
}
