package basicfs

import (
	"context"
	"fmt"
	"os"
	"path/filepath"
	"sync"

	"fil_integrate/build/cid"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/build/storiface"
)

type sectorFile struct {
	abi.SectorID
	storiface.SectorFileType
}

type Manager struct {
	Root string

	lk         sync.Mutex
	waitSector map[sectorFile]chan struct{}
	waitPiece map[cid.Commit]chan struct{}
}

func (b *Manager) GetRoot() string {
	return b.Root
}

func (b *Manager) AcquireSector(ctx context.Context, id storage.SectorRef, existing storiface.SectorFileType, allocate storiface.SectorFileType) (storiface.SectorPaths, func(), error) {
	if err := os.Mkdir(filepath.Join(b.Root, storiface.FTUnsealed.String()), 0755); err != nil && !os.IsExist(err) { // nolint
		return storiface.SectorPaths{}, nil, err
	}
	if err := os.Mkdir(filepath.Join(b.Root, storiface.FTSealed.String()), 0755); err != nil && !os.IsExist(err) { // nolint
		return storiface.SectorPaths{}, nil, err
	}
	if err := os.Mkdir(filepath.Join(b.Root, storiface.FTCache.String()), 0755); err != nil && !os.IsExist(err) { // nolint
		return storiface.SectorPaths{}, nil, err
	}

	done := func() {}

	out := storiface.SectorPaths{
	}

	for _, fileType := range storiface.PathTypes {
		if !existing.Has(fileType) && !allocate.Has(fileType) {
			continue
		}

		b.lk.Lock()
		if b.waitSector == nil {
			b.waitSector = map[sectorFile]chan struct{}{}
		}
		ch, found := b.waitSector[sectorFile{id.ID, fileType}]
		if !found {
			ch = make(chan struct{}, 1)
			b.waitSector[sectorFile{id.ID, fileType}] = ch
		}
		b.lk.Unlock()

		select {
		case ch <- struct{}{}:
		case <-ctx.Done():
			done()
			return storiface.SectorPaths{}, nil, ctx.Err()
		}

		path := filepath.Join(b.Root, fileType.String(), storiface.SectorName(id.ID))

		prevDone := done
		done = func() {
			prevDone()
			<-ch
		}

		if !allocate.Has(fileType) {
			if _, err := os.Stat(path); os.IsNotExist(err) {
				done()
				fmt.Println(path)
				return storiface.SectorPaths{}, nil, storiface.ErrSectorNotFound
			}
		}

		storiface.SetPathByType(&out, fileType, path)
	}

	return out, done, nil
}

func (b *Manager) AcquirePiece(ctx context.Context, id cid.Commit, existing storiface.SectorFileType, allocate storiface.SectorFileType) (storiface.SectorPaths, func(), error) {
	if err := os.Mkdir(filepath.Join(b.Root, storiface.FTPiece.String()), 0755); err != nil && !os.IsExist(err) { // nolint
		return storiface.SectorPaths{}, nil, err
	}

	done := func() {}

	out := storiface.SectorPaths{
	}

	for _, fileType := range storiface.PathTypes {
		if !existing.Has(fileType) && !allocate.Has(fileType) {
			continue
		}

		b.lk.Lock()
		if b.waitPiece == nil {
			b.waitPiece = map[cid.Commit]chan struct{}{}
		}
		ch, found := b.waitPiece[id]
		if !found {
			ch = make(chan struct{}, 1)
			b.waitPiece[id] = ch
		}
		b.lk.Unlock()

		select {
		case ch <- struct{}{}:
		case <-ctx.Done():
			done()
			return storiface.SectorPaths{}, nil, ctx.Err()
		}

		path := filepath.Join(b.Root, fileType.String(), storiface.PieceName(id))

		prevDone := done
		done = func() {
			prevDone()
			<-ch
		}

		if !allocate.Has(fileType) {
			if _, err := os.Stat(path); os.IsNotExist(err) {
				done()
				fmt.Println(path)
				return storiface.SectorPaths{}, nil, storiface.ErrPieceNotFound
			}
		}

		storiface.SetPathByType(&out, fileType, path)
	}

	return out, done, nil
}
