package main

import (
	"os"

	"fil_integrate/build/state-types/abi"
	"github.com/docker/go-units"
	logging "github.com/ipfs/go-log/v2"
	"github.com/urfave/cli/v2"

	"fil_integrate/seal"
)

var log = logging.Logger("bench")

func main() {
	logging.SetLogLevel("*", "INFO")

	log.Info("Starting bench")

	app := &cli.App{
		Name:    "bench",
		Usage:   "Benchmark performance of seal and window-post",
		Version: "1.0.1",
		Commands: []*cli.Command{
			testSealAndWindowPoSt,
			testSealCmd,
			testSplitDataCmd,
			testCmd,
		},
	}

	if err := app.Run(os.Args); err != nil {
		log.Warnf("%+v", err)
		return
	}
}

var testSealAndWindowPoSt = &cli.Command{
	Name:  "test-all",
	Usage: "Test Seal the sectors and generate window post",
	Flags: []cli.Flag{
		&cli.StringFlag{
			Name:  "sector-size",
			Value: "8MiB",
			Usage: "size of the sectors in bytes",
		},
		&cli.IntFlag{
			Name:  "num-agg",
			Value: 8,
			Usage: "How many window-post proofs used to aggregate",
		},
	},
	Action: func(c *cli.Context) error {
		sectorSizeInt, err := units.RAMInBytes(c.String("sector-size"))
		if err != nil {
			return err
		}
		sectorSize := abi.SectorSize(sectorSizeInt)
		numAggregate := c.Int("num-agg")
		err = seal.TestSealAndWindowPoSt(sectorSize, numAggregate)
		if err != nil {
			return err
		}
		return nil
	},
}

var testSealCmd = &cli.Command{
	Name:  "test-seal",
	Usage: "Test sealing the sectors",
	Action: func(c *cli.Context) error {
		// Test 8MiB sector
		err := seal.TestSealAndUnseal()
		if err != nil {
			return err
		}
		return nil
	},
}

var testCmd = &cli.Command{
	Name:  "test",
	Usage: "Test",
	Action: func(c *cli.Context) error {
		// Test 8MiB sector
		err := seal.Test()
		if err != nil {
			return err
		}
		return nil
	},
}

var testSplitDataCmd = &cli.Command{
	Name:  "test-split",
	Usage: "Test encode data into pieces",
	Flags: []cli.Flag{
		&cli.StringFlag{
			Name:  "sector-size",
			Value: "8MiB",
			Usage: "size of the sectors in bytes",
		},
		&cli.StringFlag{
			Name:  "data-size",
			Value: "256MiB",
			Usage: "size of the input file in bytes",
		},
	},
	Action: func(c *cli.Context) error {
		// Test 8MiB sector
		dataSize, err := units.RAMInBytes(c.String("data-size"))
		if err != nil {
			return err
		}
		sectorSizeInt, err := units.RAMInBytes(c.String("sector-size"))
		if err != nil {
			return err
		}
		sectorSize := abi.SectorSize(sectorSizeInt)
		err = seal.TestSplitDataInToPieces(sectorSize, uint64(dataSize))
		if err != nil {
			return err
		}
		return nil
	},
}
