package user

import (
	"context"
	"fmt"
	"io"
	"math/rand"
	"os"
	"path/filepath"
	"time"

	logging "github.com/ipfs/go-log/v2"
	"github.com/minio/md5-simd"
	"golang.org/x/xerrors"

	"fil_integrate/actor/connect"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/seal"
	"fil_integrate/seal/basicfs"
)

var log = logging.Logger("User")

type Range struct {
	offset uint64
	size   uint64
}

type pieceRead struct {
	piece      abi.PieceInfo
	idx        int
	pieceRange Range
}

func RunUser(ctx context.Context, conn *connect.Connection, root string, numFile int) error {
	root = filepath.Join(root, "user")
	if err := os.MkdirAll(root, 0775); err != nil {
		return err
	}
	sbfs := &basicfs.Manager{
		Root: root,
	}
	sp := seal.NewEncoder(sbfs)
	u := New(sp, conn)

	sectorSize := storage.SectorSize32MiB

	var range2Read []Range = []Range{Range{0, 0}, Range{1024, sectorSize}, Range{1024, 3*sectorSize + 1024}, Range{2*sectorSize + 2048, sectorSize + 4096}}
	var PiecesRange []pieceRead
	var err error
	b := []byte("random data")
	for i := 0; i < numFile; i++ {
		log.Infof("Generating random data")
		filename := filepath.Join(root, fmt.Sprintf("input-%d.dat", i))
		r := rand.New(rand.NewSource(time.Now().UnixNano()))
		dataSize := uint64(r.Int63n(int64(sectorSize * 6))) + 2*sectorSize
		b, err = seal.GenerateRandomData(filename, dataSize, b)
		if err != nil {
			return err
		}

		in, err := os.OpenFile(filename, os.O_RDONLY, 0644)
		if err != nil {
			return err
		}
		defer in.Close()

		finalPiece, err := u.EncodeDataToPieces(ctx, in)
		if err != nil {
			return err
		}

		for _, r := range range2Read {
			PiecesRange = append(PiecesRange, pieceRead{
				piece:      finalPiece,
				idx:        i,
				pieceRange: r,
			})
		}
		PiecesRange = append(PiecesRange, pieceRead{
			piece: finalPiece,
			idx:   i,
			pieceRange: Range{
				offset: dataSize - 1024,
				size:   1024,
			},
		})
	}
	err = conn.SendEncodeDone(ctx)
	if err != nil {
		return err
	}
	op, _, err := conn.U2PMessage(ctx)
	if err != nil {
		return err
	}
	if op != connect.OP_SEAL_DONE {
		return xerrors.Errorf("Unexpected operator")
	}

	for _, r := range PiecesRange {
		input := filepath.Join(root, fmt.Sprintf("input-%d.dat", r.idx))
		output := filepath.Join(root, "output")

		if _, err = os.Stat(output); !os.IsNotExist(err) {
			os.Remove(output)
		}
		f, err := os.OpenFile(output, os.O_WRONLY|os.O_CREATE, 0644)
		if err != nil {
			return err
		}
		err = u.ReadPieceRange(ctx, f, r.piece, r.pieceRange.offset, r.pieceRange.size)
		if err != nil {
			return err
		}
		f.Close()

		ok, err := checkDecodedFile(input, output, r.pieceRange.offset, r.pieceRange.size)
		if err != nil {
			return err
		}
		if !ok {
			fmt.Println("User: decode pieces failed")
		} else {
			fmt.Println("User: decode pieces success")
		}
	}
	return nil
}

func checkDecodedFile(file1 string, file2 string, offset uint64, size uint64) (bool, error) {
	in, err := os.Open(file1)
	if err != nil {
		return false, err
	}
	defer in.Close()
	var f1 io.Reader = in
	if offset != 0 || size != 0 {
		if _, err := in.Seek(int64(offset), io.SeekStart); err != nil {
			return false, xerrors.Errorf("seek to trailer start: %w", err)
		}
		f1 = io.LimitReader(in, int64(size))
	}

	f2, err := os.Open(file2)
	if err != nil {
		return false, err
	}
	defer f2.Close()

	inBuf := make([]byte, 2<<20)
	outBuf := make([]byte, 2<<20)

	server1 := md5simd.NewServer()
	defer server1.Close()
	server2 := md5simd.NewServer()
	defer server2.Close()

	h1 := server1.NewHash()
	defer h1.Close()
	h2 := server2.NewHash()
	defer h2.Close()

	for {
		_, inerr := f1.Read(inBuf[:])
		if err != nil && err != io.EOF {
			return false, err
		}

		_, outerr := f2.Read(outBuf[:])
		if err != nil && err != io.EOF {
			return false, err
		}

		h1.Write(inBuf)
		h2.Write(outBuf)

		if inerr == io.EOF && outerr == io.EOF {
			hash1 := h1.Sum(nil)
			hash2 := h2.Sum(nil)
			if string(hash1) != string(hash2) {
				return false, xerrors.Errorf("the output can't match input file")
			}
			break
		}
	}
	return true, nil
}
