package user

import (
	"context"
	"io"

	"golang.org/x/xerrors"

	"fil_integrate/actor/connect"
	"fil_integrate/build/cid"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/seal"
	"fil_integrate/seal/basicpiece"
)

type User struct {
	sectorSize abi.SectorSize
	encoder    seal.PieceEncoder
	conn       *connect.Connection
	cid2sidMap map[cid.Commit]abi.ActorID
}

var _ UserAPI = &User{}

func New(encoder seal.PieceEncoder, conn *connect.Connection) *User {
	u := &User{
		sectorSize: abi.SectorSize(storage.SectorSize32MiB),
		cid2sidMap: make(map[cid.Commit]abi.ActorID),
		encoder:    encoder,
		conn:       conn,
	}
	return u
}

func (u *User) EncodeDataToPieces(ctx context.Context, file storage.Data) (abi.PieceInfo, error) {
	finalPiece, pieces, err := u.encoder.EncodeDataToPieces(ctx, u.sectorSize, file)
	// send piece to provider
	for _, piece := range pieces {
		buf, err := u.encoder.LoadPiece(ctx, piece.PieceCID)
		if err != nil {
			return abi.PieceInfo{}, err
		}
		u.conn.SendPiece(ctx, buf, piece.PieceCID)
	}
	return finalPiece, err
}

func (u *User) ReadPieceRange(
	ctx context.Context,
	out io.Writer,
	piece abi.PieceInfo,
	offset uint64,
	size uint64,
) error {
	log.Infof("Reading Piece [%d:%d]", offset, offset+size)
	UnpaddedSectorSize := abi.PaddedPieceSize(u.sectorSize).Unpadded()
	DataLen := uint32(UnpaddedSectorSize) - seal.TagLen

	data, err := u.getPiece(ctx, piece.PieceCID)
	if err != nil {
		return err
	}
	piecesCommit := data.PieceCommit

	for data.HasPre {
		data, err = u.getPiece(ctx, data.PrePieceCommit)
		if err != nil {
			return err
		}
		piecesCommit = append(data.PieceCommit, piecesCommit...)
	}
	buf := data.Data[:]
	maxSize := uint64(len(piecesCommit))*uint64(DataLen) + uint64(len(buf))

	if offset == 0 && size == 0 {
		size = maxSize
	}
	if size+offset > maxSize {
		return xerrors.Errorf("Piece Size is Out of Range [offset: %w, size:%w, max_size:%w]", offset, size, maxSize)
	}

	piecesCommit = piecesCommit[offset/uint64(DataLen):]
	rangePiece := &RangePiece{
		offset: offset,
		size:   size,
		lenth:  uint64(DataLen),
	}
	for {
		rstart, rsize := rangePiece.nextRange()
		if rsize == 0 {
			break
		}

		var wbuf []byte
		if len(piecesCommit) != 0 {
			data, err := u.getPiece(ctx, piecesCommit[0])
			if err != nil {
				return err
			}

			wbuf = data.Data[rstart:]
			piecesCommit = piecesCommit[1:]
		} else {
			wbuf = buf[rstart:]
		}

		_, err = out.Write(wbuf[:rsize])
		if err != nil {
			return err
		}
	}

	return nil
}

func (u *User) getPiece(ctx context.Context, pieceCommit cid.Commit) (*basicpiece.DecodedData, error) {
	// todo: GET from chian/provider
	// miner, ok := cid2sidMap[pieceCommit]
	buf, err := u.GetPieceFromProvider(ctx, 10000, pieceCommit)
	if err != nil {
		return nil, err
	}
	return u.encoder.DecodePiece(ctx, buf)
}

func (u *User) GetPieceFromProvider(ctx context.Context, miner abi.ActorID, pieceCommit cid.Commit) ([]byte, error) {
	var buf []byte
	err := u.conn.RequestPiece(ctx, pieceCommit)
	if err != nil {
		return nil, err
	}

	op, data, err := u.conn.U2PMessage(ctx)
	if err != nil {
		return nil, err
	}
	if op != connect.OP_SEND_PIECE {
		return nil, xerrors.Errorf("Unexpected operator")
	}

	switch data.(type) {
	case connect.PieceInfo:
		buf = data.(connect.PieceInfo).Data
	default:
		return nil, xerrors.Errorf("Unexpected data")
	}
	return buf, nil
}

type RangePiece struct {
	offset uint64
	size   uint64
	lenth  uint64
}

func (rp *RangePiece) nextRange() (uint64, uint64) {
	start := rp.offset % rp.lenth
	size := min(rp.size, rp.lenth-start)
	rp.offset = rp.offset + size
	rp.size = rp.size - size
	return start, size
}

func min(x, y uint64) uint64 {
	if x < y {
		return x
	}
	return y
}
