package actor

import(
	"context"
	"os"
	"sync"
	"fmt"
	"io/ioutil"

	"golang.org/x/xerrors"
	"github.com/mitchellh/go-homedir"

	"fil_integrate/actor/connect"
	"fil_integrate/actor/user"
	"fil_integrate/actor/provider"
	"fil_integrate/actor/keeper"
)

func TestActor() error {
	conn := connect.NewConnection()
	ctx := context.Background()
    ctx, cancel := context.WithCancel(ctx)

    sdir, err := homedir.Expand("~/tmp/bench")
	if err != nil {
		return err
	}

	err = os.MkdirAll(sdir, 0775) //nolint:gosec
	if err != nil {
		return xerrors.Errorf("creating sectorbuilder dir: %w", err)
	}

	tsdir, err := ioutil.TempDir(sdir, "bench")
	if err != nil {
		return err
	}
	defer os.RemoveAll(tsdir)

	var wg sync.WaitGroup
	wg.Add(3)
	go func() {
		defer wg.Done()
		err := user.RunUser(ctx, conn, tsdir, 4)
		cancel()
		if err != nil {
			fmt.Printf("User: %+v\n", err)
			return
		}
		fmt.Println("User: done")
	} ()
	go func() {
		defer wg.Done()
		err := provider.RunProvider(ctx, conn, tsdir)
		if err != nil {
			cancel()
			fmt.Printf("Provider: %+v\n", err)
			return
		}
		fmt.Println("Provider: done")
	} ()
	go func() {
		defer wg.Done()
		err := keeper.RunKeeper(ctx, conn)
		if err != nil {
			cancel()
			fmt.Printf("Keeper: %+v\n", err)
			return
		}
		fmt.Println("Keeper: done")
	} ()
	wg.Wait()
	return nil
}