package provider

import (
	"bytes"
	"context"
	"fmt"
	"os"
	"path/filepath"

	logging "github.com/ipfs/go-log/v2"
	"golang.org/x/xerrors"

	"fil_integrate/actor/connect"
	"fil_integrate/build/cid"
	spproof "fil_integrate/build/proof"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/seal"
	"fil_integrate/seal/basicfs"
)

var log = logging.Logger("Provider")

func RunProvider(ctx context.Context, conn *connect.Connection, root string) error {
	root = filepath.Join(root, "provider")
	if err := os.MkdirAll(root, 0775); err != nil {
		return err
	}
	sbfs := &basicfs.Manager{
		Root: root,
	}
	sb := seal.NewSealer(sbfs)
	p := New(sb, 10000)

	for {
		op, data, err := conn.U2PMessage(ctx)
		if err != nil {
			return err
		}
		if op == connect.OP_ENCODE_DONE {
			break
		}

		piece := data.(connect.PieceInfo)
		err = p.SavePiece(ctx, abi.PieceInfo{
			Size:     abi.UnpaddedPieceSize(len(piece.Data)).Padded(),
			PieceCID: piece.PieceCID,
		}, bytes.NewReader(piece.Data))
		if err != nil {
			return err
		}
	}

	var perr error
	var sectorsInfo [][]storage.SectorRef
	var commitsInfo [][]storage.SectorCids
	var randomnesses []abi.PoStRandomness
	var postProofs []spproof.PoStProof
	for {
		var sectors []abi.SectorID
		var sids []storage.SectorRef
		var seeds []abi.InteractiveSealRandomness
		var proofs []spproof.Proof
		var sectorCommits []storage.SectorCids
		for i := 0; i < 4; i++ {
			sid := p.NextSectorID()

			perr = p.AddPiece(ctx, sid)
			if perr == seal.PicesNotEnoughError {
				break
			} else if perr != nil {
				return perr
			}

			cids, err := p.Sealed(ctx, sid)
			if err != nil {
				return err
			}

			err = conn.RequestSealRandom(ctx, sid.ID)
			if err != nil {
				return err
			}
			_, data, err := conn.P2KMessage(ctx)
			if err != nil {
				return err
			}
			seed := data.(abi.InteractiveSealRandomness)

			proof, err := p.GenerateCommitProof(ctx, sid, cids, seed)
			if err != nil {
				return err
			}

			err = conn.SendSealProof(ctx, proof, sid.ID, cids)
			if err != nil {
				return err
			}

			sectorCommits = append(sectorCommits, cids)
			sids = append(sids, sid)
			sectors = append(sectors, sid.ID)
			seeds = append(seeds, seed)
			proofs = append(proofs, proof)
		}
		if perr != nil {
			fmt.Println(perr.Error())
			break
		}

		proof, err := p.AggregateSealProofs(ctx, sids, sectorCommits, seeds, proofs)
		if err != nil {
			return err
		}

		err = conn.SendAggregateSealProof(ctx, proof, sectors, sectorCommits)
		if err != nil {
			return err
		}

		err = conn.RequestPoStRandom(ctx, sectors)
		if err != nil {
			return err
		}
		_, data, err := conn.P2KMessage(ctx)
		if err != nil {
			return err
		}
		seed := data.(abi.PoStRandomness)

		postProof, _, err := p.GenerateWindowPoStProofs(ctx, sids, sectorCommits, seed)
		if err != nil {
			return err
		}

		err = conn.SendWindowPoStProof(ctx, postProof, seed, p.MinerID())

		randomnesses = append(randomnesses, seed)
		sectorsInfo = append(sectorsInfo, sids)
		commitsInfo = append(commitsInfo, sectorCommits)
		postProofs = append(postProofs, postProof)
	}
	log.Infof("Aggregating window post")
	proof, err := p.AggregateWindowPoStProofs(ctx, sectorsInfo, commitsInfo, randomnesses, postProofs)
	err = conn.SendAggregateWindowPoStProof(ctx, proof, randomnesses, p.MinerID())
	if err != nil {
		return err
	}
	err = conn.SendSealDone(ctx)
	if err != nil {
		return err
	}

	for {
		op, data, err := conn.U2PMessage(ctx)
		if err != nil {
			return err
		}
		if op != connect.OP_REQUEST_PIECE {
			return xerrors.Errorf("Unexpected operator")
		}
		piece := data.(cid.Commit)

		buf, err := p.ReadPiece(ctx, piece)
		if err != nil {
			return err
		}
		err = conn.SendPiece(ctx, buf, piece)
		if err != nil {
			return err
		}
	}
	return nil
}
