package provider

import (
	"context"

	"golang.org/x/xerrors"

	"fil_integrate/build"
	"fil_integrate/build/cid"
	spproof "fil_integrate/build/proof"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/seal"
)

type Provider struct {
	sealer       seal.SectorSealer
	minerID      abi.ActorID
	sortedPieces []storage.Piece
	sectorSize   abi.SectorSize
	sectorNumber abi.SectorNumber
	// pieceID -> sector[start:end]
	pieceMap map[cid.Commit]storage.RangeSector
	// sectorID -> []pieceID
	sectorMap map[abi.SectorID][]abi.PieceInfo
}

var _ ProviderAPI = &Provider{}

func New(sealer seal.SectorSealer, miner abi.ActorID) *Provider {
	p := &Provider{
		sealer:       sealer,
		minerID:      miner,
		sectorSize:   abi.SectorSize(storage.SectorSize32MiB),
		sectorNumber: 0,
		pieceMap:     make(map[cid.Commit]storage.RangeSector),
		sectorMap:    make(map[abi.SectorID][]abi.PieceInfo),
	}
	return p
}

func (p *Provider) SavePiece(ctx context.Context, piece abi.PieceInfo, in storage.Data) error {
	return p.sealer.SavePiece(ctx, piece, in)
}

func (p *Provider) NextSectorID() storage.SectorRef {
	sid := storage.SectorRef{
		ID: abi.SectorID{
			Miner:  p.minerID,
			Number: p.sectorNumber,
		},
		ProofType: build.Spt(p.sectorSize),
	}
	p.sectorNumber++
	return sid
}

func (p *Provider) MinerID() abi.ActorID {
	return p.minerID
}

func (p *Provider) AddPiece(ctx context.Context, sid storage.SectorRef) error {
	pieces, err := p.sealer.AddPiece(ctx, sid)
	if err != nil {
		return err
	}
	p.sectorMap[sid.ID] = pieces

	return nil
}

func (p *Provider) Sealed(ctx context.Context, sid storage.SectorRef) (storage.SectorCids, error) {
	pieces, ok := p.sectorMap[sid.ID]
	if !ok {
		return storage.SectorCids{}, xerrors.Errorf("can't find the pieces info")
	}
	cids, err := p.sealer.Sealed(ctx, sid, pieces)
	if err != nil {
		return storage.SectorCids{}, err
	}

	// Store the mapping relations, pieceID -> sector[start:end]
	var offset abi.UnpaddedPieceSize
	for _, piece := range pieces {
		p.pieceMap[piece.PieceCID] = storage.RangeSector{
			Sector:   sid,
			Unsealed: cids.Unsealed,
			Offset:   abi.UnpaddedByteIndex(offset),
			Size:     piece.Size.Unpadded(),
		}
		offset += piece.Size.Unpadded()
	}

	return cids, nil
}

func (p *Provider) ReadPiece(ctx context.Context, pieceID cid.Commit) ([]byte, error) {
	sRange, ok := p.pieceMap[pieceID]
	if !ok {
		return p.sealer.ReadPiece(ctx, pieceID)
	}

	return p.sealer.UnsealedPieceRange(ctx, sRange.Sector, sRange.Unsealed, sRange.Offset, sRange.Size)
}

func (p *Provider) GenerateCommitProof(
	ctx context.Context,
	sid storage.SectorRef,
	commit storage.SectorCids,
	seed abi.InteractiveSealRandomness,
) (spproof.Proof, error) {
	pieces, ok := p.sectorMap[sid.ID]
	if !ok {
		return spproof.Proof{}, xerrors.Errorf("can't find the pieces info")
	}

	return p.sealer.GenerateCommitProof(ctx, sid, seed, pieces, commit)
}

func (p *Provider) AggregateSealProofs(
	ctx context.Context,
	sids []storage.SectorRef,
	commits []storage.SectorCids,
	seeds []abi.InteractiveSealRandomness,
	proofs []spproof.Proof,
) (spproof.Proof, error) {
	var infos []spproof.AggregateSealVerifyInfo
	for i, sid := range sids {
		infos = append(infos, spproof.AggregateSealVerifyInfo{
			Number:                sid.ID.Number,
			InteractiveRandomness: seeds[i],
			SealedCID:             commits[i].Sealed,
			UnsealedCID:           commits[i].Unsealed,
		})
	}
	return p.sealer.AggregateSealProofs(spproof.AggregateSealVerifyProofAndInfos{
		SealType:      build.Spt(p.sectorSize),
		AggregateType: abi.DefaultAggregationType(),
		Infos:         infos,
	}, proofs)
}

func (p *Provider) GenerateWindowPoStProofs(
	ctx context.Context,
	sids []storage.SectorRef,
	commits []storage.SectorCids,
	randomness abi.PoStRandomness,
) (spproof.PoStProof, []abi.SectorID, error) {
	var challengedSectors []spproof.SectorInfo
	if len(sids) != len(commits) {
		return spproof.PoStProof{}, nil, xerrors.Errorf("can't use %d sector and %d commitment to gengerate window post", len(sids), len(commits))
	}
	for i, sid := range sids {
		challengedSectors = append(challengedSectors, spproof.SectorInfo{
			SealType:     build.Spt(p.sectorSize),
			SectorNumber: sid.ID.Number,
			SealedCID:    commits[i].Sealed,
		})
	}
	return p.sealer.GenerateWindowPoStProofs(ctx, p.minerID, challengedSectors, randomness)
}

func (p *Provider) AggregateWindowPoStProofs(
	ctx context.Context,
	sidsArr [][]storage.SectorRef,
	commitsArr [][]storage.SectorCids,
	randomnesses []abi.PoStRandomness,
	proofs []spproof.PoStProof,
) (spproof.PoStProof, error) {
	var challengedSectors []spproof.SectorInfo
	var sectorCount []uint
	var srandomnesses []abi.PoStRandomness

	for i, sids := range sidsArr {
		for j, sid := range sids {
			challengedSectors = append(challengedSectors, spproof.SectorInfo{
				SealType:     build.Spt(p.sectorSize),
				SectorNumber: sid.ID.Number,
				SealedCID:    commitsArr[i][j].Sealed,
			})
		}
		sectorCount = append(sectorCount, uint(len(sids)))
	}

	srandomnesses = append(srandomnesses, randomnesses...)

	return p.sealer.AggregateWindowPoStProofs(spproof.AggregateWindowPostInfos{
		PoStType:          proofs[0].PoStProof,
		AggregateType:     abi.DefaultAggregationType(),
		ChallengedSectors: challengedSectors,
		SectorCount:       sectorCount,
		Randomnesses:      srandomnesses,
		Prover:            p.minerID,
	}, proofs)
}
