package keeper

import (
	"context"
	// "fmt"

	logging "github.com/ipfs/go-log/v2"
	"github.com/minio/blake2b-simd"
	"golang.org/x/xerrors"

	"fil_integrate/actor/connect"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/seal"
)

var log = logging.Logger("Keeper")

func RunKeeper(ctx context.Context, conn *connect.Connection) error {
	v := seal.NewVerifier()
	kp := New(v)
	var random = blake2b.Sum256([]byte("keeper"))
	srm := make(map[abi.SectorID]abi.InteractiveSealRandomness)
	prm := make(map[abi.PoStRandomness][]abi.SectorID)

	for {
		op, data, err := conn.P2KMessage(ctx)
		if err != nil {
			return err
		}
		switch op {
		case connect.OP_REQUEST_SEAL_RANDOM:
			log.Infof("Generating seal random seed")

			sid := data.(abi.SectorID)
			random := blake2b.Sum256(random[:])
			// random = seed
			srm[sid] = random
			err := conn.SendSealRandom(ctx, random)
			if err != nil {
				return err
			}
		case connect.OP_REQUEST_POST_RANDOM:
			log.Infof("Generating post random seed")
			sids := data.([]abi.SectorID)
			random = blake2b.Sum256(random[:])
			prm[random] = sids
			err := conn.SendPoStRandom(ctx, random)
			if err != nil {
				return err
			}
		case connect.OP_SEND_SEAL_PROOF:
			info := data.(connect.SealProofInfo)
			seed, ok := srm[info.Sector]
			if !ok {
				return xerrors.Errorf("can't find the random seed", info.Sector)
			}
			ok, err := kp.VerifySeal(ctx, info.Sector, seed, info.Commit, info.Proof)
			if err != nil {
				return err
			}
			if !ok {
				return xerrors.Errorf("Verify Seal proof failed")
			}
		case connect.OP_SEND_AGGREGATE_SEAL_PROOF:
			info := data.(connect.AggregateSealProofInfo)
			var seeds []abi.InteractiveSealRandomness
			for _, sid := range info.Sectors {
				seed, ok := srm[sid]
				if !ok {
					return xerrors.Errorf("can't find the random seed", sid)
				}
				seeds = append(seeds, seed)
			}
			ok, err := kp.VerifyAggregateSeals(ctx, info.Sectors, seeds, info.Commits, info.Proof)
			if err != nil {
				return err
			}
			if !ok {
				log.Warnf("Keeper: Verify Seal Aggregation proof failed")
			}
		case connect.OP_SEND_WINDOW_POST_PROOF:
			log.Infof("verifying window post proof")
			info := data.(connect.WindowPoStProofInfo)
			sids, ok := prm[info.Randomness]
			if !ok {
				return xerrors.Errorf("can't find the random seed", info.Randomness)
			}
			ok, err := kp.VerifyWindowPoSt(ctx, sids, info.Randomness, info.Proof)
			if err != nil {
				return err
			}
			if !ok {
				return xerrors.Errorf("Verify Window-PoSt proof failed")
			}
		case connect.OP_SEND_AGGREGATE_WINDOW_POST_PROOF:
			log.Infof("verifying aggregate window post proof")
			info := data.(connect.AggregateWindowPoStProofInfo)
			var sidsArr [][]abi.SectorID
			for _, random := range info.Randomnesses {
				sids, ok := prm[random]
				if !ok {
					log.Warnf("can't find the random seed", random)
				}
				sidsArr = append(sidsArr, sids)
			}
			ok, err := kp.VerifyAggregateWindowPostProofs(ctx, sidsArr, info.Randomnesses, info.Proof)
			if err != nil {
				return err
			}
			if !ok {
				return xerrors.Errorf("Verify Window-PoSt proof failed")
			}
			return nil
		default:
			return xerrors.Errorf("unsupported operator")
		}
	}
	return nil
}
