package keeper

import (
	"context"

	"golang.org/x/xerrors"

	"fil_integrate/build"
	"fil_integrate/build/cid"
	spproof "fil_integrate/build/proof"
	"fil_integrate/build/state-types/abi"
	"fil_integrate/build/storage"
	"fil_integrate/seal"
)

type Keeper struct {
	verifier   seal.SectorVerifier
	sectorSize abi.SectorSize
	// sid -> (commd, commr)
	commMap map[abi.SectorID]storage.SectorCids
}

var _ KeeperAPI = &Keeper{}

func New(verifier seal.SectorVerifier) *Keeper {
	kp := &Keeper{
		verifier:   verifier,
		sectorSize: abi.SectorSize(storage.SectorSize32MiB),
		commMap:    make(map[abi.SectorID]storage.SectorCids),
	}
	return kp
}

func (k *Keeper) VerifySeal(
	ctx context.Context,
	sid abi.SectorID,
	randomness abi.InteractiveSealRandomness,
	commit storage.SectorCids,
	proof spproof.Proof,
) (bool, error) {
	log.Infof("[%d] Verifying Seal Sector", sid.Number)
	ok, err := k.verifier.VerifySeal(spproof.SealVerifyInfo{
		SealType:              build.Spt(k.sectorSize),
		Miner:                 sid.Miner,
		Number:                sid.Number,
		InteractiveRandomness: randomness,
		SealProof:             proof,
		SealedCID:             commit.Sealed,
		UnsealedCID:           commit.Unsealed,
	})

	if ok && err == nil {
		k.commMap[sid] = commit
	}

	return ok, err
}

func (k *Keeper) VerifyAggregateSeals(
	ctx context.Context,
	sids []abi.SectorID,
	randomnesses []abi.InteractiveSealRandomness,
	commits []storage.SectorCids,
	proof spproof.Proof,
) (bool, error) {
	log.Infof("verifying aggregate seal proof")
	infos := make([]spproof.AggregateSealVerifyInfo, len(sids))
	if len(sids) != len(randomnesses) || len(sids) != len(commits) {
		return false, xerrors.Errorf("the lenth of the seal infos don't match")
	}
	for i := 0; i < len(infos); i++ {
		infos[i] = spproof.AggregateSealVerifyInfo{
			Number:                sids[i].Number,
			InteractiveRandomness: randomnesses[i],
			SealedCID:             commits[i].Sealed,
			UnsealedCID:           commits[i].Unsealed,
		}
	}

	ok, err := k.verifier.VerifyAggregateSeals(spproof.AggregateSealVerifyProofAndInfos{
		Miner:          sids[0].Miner,
		SealType:       build.Spt(k.sectorSize),
		AggregateType:  abi.DefaultAggregationType(),
		AggregateProof: proof,
		Infos:          infos,
	})

	if ok && err == nil {
		for i, sid := range sids {
			k.commMap[sid] = commits[i]
		}
	}

	return ok, err
}

func (k *Keeper) VerifyWindowPoSt(
	ctx context.Context,
	sids []abi.SectorID,
	randomness abi.PoStRandomness,
	proof spproof.PoStProof,
) (bool, error) {
	var sectors []spproof.SectorInfo
	for _, sid := range sids {
		commit, ok := k.commMap[sid]
		if !ok {
			return false, xerrors.Errorf("Sector:%+v not found", sid)
		}
		sectors = append(sectors, spproof.SectorInfo{
			SealType:     build.Spt(k.sectorSize),
			SectorNumber: sid.Number,
			SealedCID:    commit.Sealed,
		})
	}

	return k.verifier.VerifyWindowPoSt(spproof.WindowPoStVerifyInfo{
		Randomness:        randomness,
		Proof:             proof,
		ChallengedSectors: sectors,
		Prover:            sids[0].Miner,
	})
}

func (k *Keeper) VerifyAggregateWindowPostProofs(
	ctx context.Context,
	sidsArr [][]abi.SectorID,
	randomnesses []abi.PoStRandomness,
	proof spproof.PoStProof,
) (bool, error) {
	var sectors []spproof.SectorInfo
	sectorCount := make([]uint, len(sidsArr))
	for i, sids := range sidsArr {
		sectorCount[i] = uint(len(sids))
		for _, sid := range sids {
			commit, ok := k.commMap[sid]
			if !ok {
				return false, xerrors.Errorf("Sector:%+v not found", sid)
			}
			sectors = append(sectors, spproof.SectorInfo{
				SealType:     build.Spt(k.sectorSize),
				SectorNumber: sid.Number,
				SealedCID:    commit.Sealed,
			})
		}
	}

	postType, err := sectors[0].SealType.RegisteredWindowPoStProof()
	if err != nil {
		return false, err
	}

	return k.verifier.VerifyAggregateWindowPostProofs(spproof.AggregateWindowPostInfos{
		PoStType:          postType,
		AggregateType:     abi.DefaultAggregationType(),
		AggregateProof:    proof,
		ChallengedSectors: sectors,
		SectorCount:       sectorCount,
		Randomnesses:      randomnesses,
		Prover:            sidsArr[0][0].Miner,
	})
}

func (k *Keeper) VerifyPieceAndDataRoot(commd cid.Commit, pieces []abi.PieceInfo) (bool, error) {
	return k.verifier.VerifyPieceAndDataRoot(build.Spt(k.sectorSize), commd, pieces)
}
